// $Id: APPSPACK_Scheduler.cpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Scheduler.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Scheduler.cpp
  \brief Implements APPSPACK::Scheduler
*/

#include "APPSPACK_Scheduler.hpp"
#include "APPSPACK_Print.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_GCI.hpp"
#include "APPSPACK_Utils.hpp"
#include "APPSPACK_Executor_Oracle.hpp"
#include "APPSPACK_Vector.hpp"
#include "APPSPACK_Mediator.hpp"
#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Executor_Oracle.hpp"

APPSPACK::Scheduler::Scheduler(int argc, char* argv[]) :
  rank(APPSPACK::GCI::init(argc, argv)),
  nprocs(APPSPACK::GCI::getNumProcs())
{
  // Verify an input file was given.
  exitIf(argc < 2, "Usage: " + (string)argv[0] + " <input file>");

  // Verify at least 3 processors were given.
  exitIf(nprocs < 2, "Error: This program requires at least 3 processes.");

  // Read in parameters files.
  paramsFile = argv[1];
  bool status = APPSPACK::parseTextInputFile(paramsFile, params);
  exitIf(status==false, "Error: Something wrong with parameter file.");

  // Now get assignments and give to all processors.
  if (rank == 0)
    sendAssignments();
  else
    recvAssignments();
}

void APPSPACK::Scheduler::init()
{
  if (isMaster() == true)
    initMaster();
  else if (isWorker() == true)
    initEvalWorker();
  else 
    initCitizenWorker();
}

APPSPACK::Parameter::List& APPSPACK::Scheduler::getParameters()
{
  return params;
}

void APPSPACK::Scheduler::cleanup()
{
}

void APPSPACK::Scheduler::getCitizenInfo(int i, string& type, string& name,
                                         vector<int>& workerRank)
{
  name="Citizen " + IntToString(i);
  type=params.sublist(name).getParameter("Type", "undefined");
  workerRank.resize(0);
  for (int i=0; i<assign.size(); i++)
  {
    if (assign[i] == name)
    {
      // Citizen "name" owns processor with rank i.
      workerRank.push_back(i);
    }
  }
}

void APPSPACK::Scheduler::masterLoop(APPSPACK::Citizen::Generator& generator)
{
  // Create linear constraints object to determine scaling vector needed by conveyor.
  Constraints::Linear linear(params.sublist("Linear"));
  // Create an executor object
  Executor::Oracle executor(nWorkers); 
  // Create conveyor used by mediator to perform function evaluations in parallel.
  Conveyor conveyor(params.sublist("Mediator"), linear.getScaling(), executor);
  // Create Mediator.
  Mediator mediator(params.sublist("Mediator"), conveyor);
  
  Combiner::Generic combiner;
  Citizen::Interface* ci;
  vector<Citizen::Interface*> citizenList;
  
  string name;
  string type;
  vector <int> workerRank;
  for (int i=1; i<=nCitizens; i++)
  {
    getCitizenInfo(i, type, name, workerRank);
    ci = generator.newCitizenType(params, linear, combiner,
                                  type, name, workerRank);
    mediator.push(*ci);
    citizenList.push_back(ci);
  }

  // Now process demands of citizens.
  mediator.mediate();
  
  // Clean up;   
  for (int i=0; i<citizenList.size(); i++)
    delete citizenList[i];

  // Make sure all points have been deleted.
  Point::printMemoryStats();

  // Send a termination command to every worker
  GCI::initSend();
  GCI::pack(1);
  for (int i = 1; i < nprocs; i ++)
  {
    GCI::send(Executor::Oracle::Terminate, i);
  }

  APPSPACK::GCI::exit();
  cout << "Master " << rank << " exiting ..." << endl;
}

void APPSPACK::Scheduler::workerLoop(APPSPACK::Evaluator::Interface& evaluator)
{
  // Continuously receive and process incoming messages
  while (1)
  {
    // Blocking receive for the next message
    int msgtag, junk;
    APPSPACK::GCI::recv();
    APPSPACK::GCI::bufinfo(msgtag, junk);
    
    // Check for termination
    if (msgtag == APPSPACK::Executor::Oracle::Terminate) 
    {
      break;
    }
    // Local vars to be packed and unpacked
    int tag;
    APPSPACK::Vector x;
    bool isF;
    APPSPACK::Vector f;
    string msg;
    // Unpack the latest message
    APPSPACK::GCI::unpack(tag);
    APPSPACK::GCI::unpack(x);
    // Evaluate the function
    evaluator(tag,x,f,msg);
    
    // Send a reply
    APPSPACK::GCI::initSend();
    APPSPACK::GCI::pack(tag);
    APPSPACK::GCI::pack(f);
    APPSPACK::GCI::pack(msg);
    APPSPACK::GCI::send(APPSPACK::Executor::Oracle::Feval,0);    
  }

  APPSPACK::GCI::exit();
  cout << "Evaluation Worker " << rank << " exiting ..." << endl;
}

void APPSPACK::Scheduler::citizenLoop(APPSPACK::Citizen::Generator& generator)
{
  // Create linear constraints object to determine scaling vector needed by conveyor.
  Constraints::Linear linear(params.sublist("Linear"));
  Combiner::Generic combiner;
  Citizen::Interface* ci;  
  string name = assign[rank];  // Citizen i.
  string type = params.sublist(name).getParameter("Type", "undefined");
  
  // Create new worker citizen of appropriate type.
  vector<int> myrank;
  myrank.push_back(rank);
  ci = generator.newCitizenType(params, linear, combiner,
                                type, name, myrank);
  
  bool doWork = true;
  while (doWork)
    doWork = ci->worker();

  // Delete worker citizen.
  delete ci;

  APPSPACK::GCI::exit();
  cout << "Citizen Worker " << rank << " exiting ..." << endl;
}

bool APPSPACK::Scheduler::isMaster() const
{
  exitIf(assign.size() != nprocs, "Assignment vector not established");
  return (assign[rank] == "Master");
}

bool APPSPACK::Scheduler::isCitizen() const
{
  exitIf(assign.size() != nprocs, "Assignment vector not established");
  return ( (assign[rank] != "Master") && (assign[rank] != "Worker") );
}

bool APPSPACK::Scheduler::isWorker() const
{
  exitIf(assign.size() != nprocs, "Assignment vector not established");
  return (assign[rank] == "Worker");
}

int APPSPACK::Scheduler::getRank() const
{
  return rank;
}

void APPSPACK::Scheduler::initMaster()
{  
  // Send the evaluator information to each worker
  APPSPACK::GCI::initSend();
  params.sublist("Evaluator").pack();
  for (int i=1; i < assign.size(); i ++)
  {
    if (assign[i] == "Worker")
    {
      APPSPACK::GCI::send(APPSPACK::Executor::Oracle::Init, i);  
    }
  }

  // Send Citizen parameters.
  for (int i=1; i < assign.size(); i ++)
  {
    if (assign[i] != "Worker")
    {
      APPSPACK::GCI::initSend();
      params.pack();
      APPSPACK::GCI::send(APPSPACK::Executor::Oracle::Init, i);  
    }
  }
}

void APPSPACK::Scheduler::initCitizenWorker()
{
  // Unpack parameter list sent by master
  APPSPACK::GCI::recv(APPSPACK::Executor::Oracle::Init,0);
  params.unpack();
}

void APPSPACK::Scheduler::initEvalWorker()
{
  // Unpack parameter list sent by master
  APPSPACK::GCI::recv(APPSPACK::Executor::Oracle::Init,0);
  params.unpack();
}

void APPSPACK::Scheduler::sendAssignments()
{
  exitIf(rank != 0, (string) "APPSPACK::Scheduler::sendAssignment() "
	 + "may only be called by master.", false);
  
  Parameter::List& medParams = params.sublist("Mediator");
  exitIf(!medParams.isParameter("Citizen Count"), 
	 "Error: \"Citizen Count\" must be set.");

  nCitizens = medParams.getParameter("Citizen Count", -1);
  exitIf(nCitizens < 1, "Error: Number of Citizens must be greater than 0.");

  // First determine assignment vector.
  assign.resize(0);
  string base="Citizen ";
  for (int i=1; i<= nCitizens; i++)
  {
    // Must have nCitizens defined starting at 1.
    string name = base + IntToString(i);
    exitIf(!params.isParameter(name), 
	   "Error: \"Citizen " + IntToString(i) + "\" must be defined.");

    // Each Citizen must specify it's type.
    Parameter::List& citizenParams = params.sublist(name);
    exitIf(!citizenParams.isParameter("Type"),
	   "Error: \"Citizen " + IntToString(i) + "\" must define \"Type\".");
    string type=citizenParams.getParameter("Type", "Undefined");

    // Citizens may request multiple processors.
    int np = citizenParams.getParameter("Processors", 0);
    for (int j=0; j<np; j++)
      assign.push_back(name);
  }
  
  // Now add in master and evaluation workers.
  // Assign.size() now equals total number of processors requested by Citizens.
  nWorkers = nprocs - assign.size() - 1;
  exitIf(nWorkers < 1, "Error: Insufficient number of processors."); 
  while  (assign.size() < nprocs-1)
    assign.insert(assign.begin(), "Worker");
  assign.insert(assign.begin(), "Master");
  
  // Send the evaluator information to each worker
  APPSPACK::GCI::initSend();
  APPSPACK::GCI::pack(assign);
  for (int i=0; i < nprocs-1; i ++)
    APPSPACK::GCI::send(APPSPACK::Executor::Oracle::Init, i+1);    
}

void APPSPACK::Scheduler::recvAssignments()
{
  exitIf(rank==0,"Master should not receive assignments");

  // Unpack assignment vector.
  APPSPACK::GCI::recv(APPSPACK::Executor::Oracle::Init,0);
  APPSPACK::GCI::unpack(assign);
}

string APPSPACK::Scheduler::IntToString(int n) const
{
  ostringstream ss;
  ss << n;
  return ss.str();
}

void APPSPACK::Scheduler::exitIf(bool check, string msg, 
				 bool masterOnly) const
{
  if (check == true)
  {
    if (masterOnly)
    {
      if (rank == 0)
      {
	cerr << msg << endl;
	throw "APPSPACK Error";
      }
    }
    else
    {
      cerr << msg << endl;
      throw "APPSPACK Error";
    }
  }
}
