// $Id: APPSPACK_Citizen_TGP.hpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_TGP.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_TGP.hpp
  \brief Class definition for APPSPACK::Citizen_TGP
*/

#ifndef APPSPACK_CITIZEN_TGP_HPP
#define APPSPACK_CITIZEN_TGP_HPP

#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Cache_Manager.hpp"

namespace APPSPACK
{

namespace Citizen
{

//! A class that exchanges points using a local system call.
/*! 
  
  These parameters are are stored in the Parameter::List specified
  by the sublist "Mediator". 
  <ul>
  <li>"Executable Name" Defaults to a.out.
  <li>"Input Prefix"  Defaults to "oracle_input".
  <li>"Output Prefix" Defaults to "oracle_output".
  <li>"Default Step"  Defaults to 1.0.
  <li>"Default Votes" Defaults to 1.
  <li>"Output File Precsion" Defaults to 14 digits of accuracy.
  </ul>
  
  \see \ref pageMediator for further details.
*/
class TGP : public Interface
{
public:

  //! Constructor 
  TGP(Parameter::List& params_in,
      const Constraints::Linear& constraints_in,
      Combiner::Generic& combiner_in, int rank,
      const string name = "");
  
  //! Destructor
  ~TGP();

  //! Prints initial state of citizen.
  virtual void preprocess();

  //! Prints final state of citizen.
  virtual void postprocess();
  
  //! Take in list of evaluated appspacks. Give list of new points.
  virtual void exchange(const ConveyorList& R, ConveyorList& Wnew,
                        const map<string, vector<int> >& ownermap);

  //! Vote on points to be evaluated.
  virtual void vote(const ConveyorList& W, 
		    const map<string, vector<int> >& ownermap,
		    vector<int>& tagOrder);

  //! Return Citizen's state.
  virtual Citizen::State getState();

  //! Returns Citizen's name.
  virtual const string& getName() const;
  
  //! Call's tgp oracle.
  virtual bool worker();
private:

  //! Creates command strings and file names.
  /*! If useTGP is true, then call TGP.  Otherwise just create LHS points.  In either case,
    create nPoints LHS points.
   */
  void createStrings(bool useTGP, int nPoints);

  //! Write new evalution to cache file evalsFileName.
  void writeNewEvals(const ConveyorList& R, const map<string, vector<int> >& ownermap);

  //! Pack new evalution to cache file evalsFileName.
  void packNewEvals();

  //! Reset new evals record.
  void resetEvalRecord();

  //! Reads numPoints from output file
  void readOutputFile(Matrix& TP, int numPoints);

  //! Adds rows of TP as points to Wnew list.
  void addTrialPoints(ConveyorList& Wnew, const Matrix& TP);
  
  //! Converts an int to a string.
  string IntToString(int n) const;

  //! Converts a string of form line = "1 2 3" to a vector x =<1 2 3>.
  void StringToVector(const string& line, Vector& x) const;

private:

  //! Name of citizen.
  const string name;

  //! Parameters
  Parameter::List& params;
  
  //! Combiner reference needed when generating points
  Combiner::Generic& combiner;

  //! Worker cache. Worker cache may use  own tolerance.
  Cache::Manager* tgpCache;

  //! Constraints
  const Constraints::Linear& constraints;

  //! Static counter used to generate unique tags
  static int staticCount;

  //! Static counter for evaluation to prevent TGP's from simultaneously running.
  static int staticEvalCount;

  //! Counts number of points evaluated.
  int pointCount;

  //! Candidate to choose point from.
  string candidateFile;

  //! Number of candidate locations requested from oracle
  int nCandidates;

  //! Number of trial points to use from candidate location list.
  int nTrialPoints;

  //! Name of executable that generates points.
  string executableName;

  //! Name of cache file to be read in by TGP oracle.
  string evalsFileName;

  //! Name of worker cache file to be read in by TGP oracle.
  string workerFileName;

  //! If false, a new cache file is made.
  bool appendEvalsFile;

  //! Stream connected to evalsFileName.
  ofstream evalsOutStream;

  //! Command prefix: evalsFileName nCandiates databounds
  string tgpCommandPrefix;

  //! They system command used.
  string tgpCommand;
  
  //! File precision.
  int precision;

  //! Command tag.
  string fileTag;
  
  //! Stores tags of points in order we want submitted.
  vector<int> queueOrder;

  //! True if tgp is currently running.
  bool tgpRunning;

  //! Default step size.
  const double defaultStep;

  //! Rank of worker that will be sending messages to TGP.
  const int workerRank;

  //! Number of initial LHS points to use in sampling.
  int initialSampleSize;

  //! Stores list of initial points.
  Matrix initialLHS;

  //! If true, all points in initial LHS have been evaluated.
  bool initialSampleComplete;

  //! Determines the minimum number of evaluations that must be called before TGP.
  int minEvaluations;

  //! Perform global or local search.
  bool useGlobal;

  //! Determine tolerance for switching to local search
  double globalTolerance;

  //! States whether or not to support noise
  bool useNoiseSupport;

  //! Used to define local neighborhood.
  double contractionFactor;

  //! Best x value point.
  Vector bestX;

  //! Best function value corresponding to bestX.
  Vector bestF;

  //! Stores function values corresponding to X since last call.
  vector<Vector> F;
  //! Stores X values corresponding to F. 
  vector<Vector> X;
  //! Stores tags values corresponding to X since last call.
  vector<int> Xtags;
  //! Stores step value corresponding to X since last call.
  Vector Xstep;
  //! Determines if TGP point or not.
  vector<string> ownerName;

  //! TGP cache tolerance
  double tgpCacheTolerance;

};

}
}
#endif
