// $Id: APPSPACK_Citizen_SystemCall.hpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_SystemCall.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_SystemCall.hpp
  \brief Class definition for APPSPACK::Citizen_SystemCall
*/

#ifndef APPSPACK_CITIZEN_SYSTEMCALL_HPP
#define APPSPACK_CITIZEN_SYSTEMCALL_HPP

#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Citizen_Interface.hpp"

namespace APPSPACK
{

namespace Citizen
{

//! A class that exchanges points using a local system call.
/*! 
  
  These parameters are are stored in the Parameter::List specified
  by the sublist "Mediator". 
  <ul>
  <li>"Executable Name" Defaults to a.out.
  <li>"Input Prefix"  Defaults to "oracle_input".
  <li>"Output Prefix" Defaults to "oracle_output".
  <li>"Default Step"  Defaults to 1.0.
  <li>"Default Votes" Defaults to 1.
  <li>"Output File Precsion" Defaults to 14 digits of accuracy.
  </ul>
  
  File input output communication is used in a similar fashion to how
  points are evaluated.  Essentially the oracle specified in the APPSPACK 
  input deck is called like
  
  \verbatim
  <Executable Name> <Input Prefix>.<tag> <Output Prefix>.<tag> <tag>
  \endverbatim
  
  For example, the call might look something like:
  
  \verbatim
  example_oracle oracle_in.27 oracle_output.27 27
  \endverbatim
  
  Here tag denotes an index that is unique to the particular call to the
  point generating executable.
  
  An example oracle input file (generated by Citizen::SystemCall) might look like

  \verbatim
  New Evaluations: 5
  f=[ 2.42e-02  ] x=[ 1.58e-01 -1.07e-02  ] own=[ false ] 
  f=[ 2.42e-02  ] x=[ 1.55e-01 -1.43e-03  ] own=[ true ] 
  f=[ 2.83e-02  ] x=[ -1.68e-01 4.35e-03  ] own=[ false ] 
  f=[ 2.30e-02  ] x=[ 1.49e-01 -1.73e-02  ] own=[ true ] 
  f=[ 2.54e-01  ] x=[ 5.04e-01 4.35e-03  ] own=[ false ] 
  \endverbatim

  This specifies to the oracle executable that 5 new points have
  returned from the evaluation queue since the last call.  Two of
  these returned evaluations, the second and the fourth, were
  submitted to the Mediator by the oracle executable.  Three of the
  returned evaluations were submitted by another Citizen.  

  A corresponding example oracle output file (generated by the
  executable specified by "Executable Name") might in turn look like

  \verbatim
  x=[ 0.13878 0.00572388 ] votes=[ 9 ] 
  x=[ 0.393462 -0.0871535 ] votes=[ 10 ] step=[ 1.5 ] 
  \endverbatim

  Here the user has specified 8 votes for the point corresponding to
  tag 56 and submits two new points with 9 and 10 votes respectively.
  Note that for the second point the user has also specified the
  (optional) corresponding step size of 1.5.  By default the
  APPSPACK::Iterator will only vote on points that it generated and
  will cast a maximum of \f$ k \f$ votes where \f$ k \f$ denotes the
  number of points currently in the queue \e that \e it \e owns.
  Hence, since the queue has 7 points, the number of votes cast for
  Iterator points will be less than 8.  By starting with 8 votes, the
  user has ensured that the points they generated will be evaluated
  next, i.e. the Mediator will shuffle these points to the front of
  the queue.

  \see \ref pageMediator for further details.
*/
class SystemCall : public Interface
{
public:

  //! Constructor 
  SystemCall(Parameter::List& params_in,
	     Combiner::Generic& combiner_in,
	     const string name = "");
  
  //! Destructor
  ~SystemCall();

  //! Prints initial state of citizen.
  virtual void preprocess();

  //! Prints final state of citizen.
  virtual void postprocess();
  
  //! Take in list of evaluated appspacks. Give list of new points.
  virtual void exchange(const ConveyorList& R, ConveyorList& Wnew,
                        const map<string, vector<int> >& ownermap);

  //! Vote on points to be evaluated.
  virtual void vote(const ConveyorList& W, 
		    const map<string, vector<int> >& ownermap,
		    map<int,int>& tagVote);

  //! Return Citizen's state.
  virtual Citizen::State getState();

  //! Returns Citizen's name.
  virtual const string& getName() const;
  
private:
  
  //! Creates command strings and file names.
  void createStrings();

  //! Writes W and R to an input file.
  void writeInputFile(const ConveyorList& R, const vector<int>& citizen_tags);
  
  //! Run executable on new points.
  void runProgram();
  
  //! Reads output file
  void readOutputFile(ConveyorList& Wnew);
  
  //! Parse a line of the form tag = [ int ] vote = [ int ]
  void processTagVote(const string& line);

  /*! \brief Parse a line of the form 
    x = [x1 x2 ... xn ] vote = [ int ] step = [ double ]
    where step is optional. \todo finish comments.
  */
  Point* processNewX(const string& line);

  //! Deletes input/output files when finished.
  void deleteFiles();  
  
  //!  Returns contents of name = [ string ].
  string getContent(const string& line, const string& name) const;
							    
  //! Convert a string to a int.
  int str2int(const string& item) const;

  //! Convert a string to a double.
  double str2dbl(const string& item) const;

  //! Convert a string to vector.
  void str2vec(const string& item, Vector& d) const;
  
private:
  
  //! Combiner reference needed when generating points
  Combiner::Generic& combiner;

  //! Static counter used to generate unique tags
  static int staticCount;

  //! Parameters
  Parameter::List& params;

  //! Name of citizen.
  string name;

  //! Prefix for input file
  string inputPrefix;

  //! File name for input file
  string inputFileName;

  //! Input file created.
  bool isInputFile;

  //! Prefix for output file
  string outputPrefix;

  //! File name for output file
  string outputFileName;

  //! Output file created.
  bool isOutputFile;

  //! Name of executable that generates points.
  string executableName;

  //! They system command used.
  string execString;
  
  //! File precision.
  int precision;

  //! Command tag.
  string fileTag;

  //! Stores tag votes from last call to exchange.
  map<int,int> tagVote;

  //! Default step size.
  const double defaultStep;

  //! Default number of votes given to a generated point.
  const int defaultVotes;
};

}
}
#endif
