// $Id: APPSPACK_Citizen_LHS.cpp,v 1.3 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_LHS.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_LHS.cpp
  \brief Implements APPSPACK::Citizen_LHS

  \todo Josh Comment parameters.
*/

#include "APPSPACK_Citizen_LHS.hpp"
#include "APPSPACK_Combiner_Generic.hpp"
#include "APPSPACK_Executor_Oracle.hpp"
#include "APPSPACK_Print.hpp"


int APPSPACK::Citizen::LHS::staticCallCount = 0;

APPSPACK::Citizen::LHS::LHS(Parameter::List& params_in,
                            const Constraints::Linear& constraints_in,
                            Combiner::Generic& combiner_in, const string name_in) :
  params(params_in.sublist(name_in)),
  name(name_in + "(LHS)"),
  combiner(combiner_in),
  constraints(constraints_in),
  sampleSize(params.getParameter("Sample Size", 100)),
  samplesFile(params.getParameter("LHS File", "LHS.txt")),
  executableName(params.getParameter("Executable Name", "a.out")),
  defaultStep(params.getParameter("Default Step", 1.0))
{
  callCount = 0;
}

APPSPACK::Citizen::LHS::~LHS()
{
}

void APPSPACK::Citizen::LHS::preprocess()
{ 
}

void APPSPACK::Citizen::LHS::postprocess()
{
}

void APPSPACK::Citizen::LHS::exchange(const APPSPACK::ConveyorList& R,
                                      APPSPACK::ConveyorList& Wnew, 
                                      const map<string, vector<int> >& ownermap)
{    
  //! Only one call to LHS is allowed.
  if (callCount++ > 0)
    return;

  // Determine lower and upper bounds in R-notation.
  //! R-notation bounds: "c(l1,12,...,ln),c(u1,u2,...,un)".
  const Vector& bLower=constraints.getLower();
  const Vector& bUpper=constraints.getUpper();

  // Write using string streams.
  ostringstream ss;

  // Process upper and lower bounds.
  ss << "\"c(" << bLower[0];
  for (int i=1; i<bLower.size(); i++)
    ss << "," << bLower[i];
  ss << "),c(" << bUpper[0];
  for (int i=1; i<bUpper.size(); i++)
    ss << "," <<bUpper[i];
  ss << ")\"";

  // Load in initial LHS points.  
  string prefix  = executableName + " " + ss.str() + " " +  samplesFile;
  string nsample = IntToString(sampleSize);
  fileTag = IntToString(staticCallCount++);
  string cmd = prefix + " " + IntToString(sampleSize) + " " + fileTag;
  cout << "Calling: " << cmd << endl;
  system(cmd.c_str());
  
  // Grab all points.
  Matrix LHS;
  readOutputFile(LHS);

  // Now add trials in LHS to Wnew;
  addTrialPoints(Wnew, LHS);
}

void APPSPACK::Citizen::LHS::vote(const ConveyorList& W, 
                                  const map<string, vector<int> >& ownermap,
                                  vector<int>& tagOrder)
{
  tagOrder.clear();

  // Get list of tags in the queue W.
  vector<int> wtags;
  W.getTagList(wtags);
  
  // Get list of tags that this owns.
  const vector<int>& lhs_tags = ownermap.find(name)->second;

  // Give all points that we own a vote equal to GSS.
  for (int i=0; i<lhs_tags.size(); i++)
  {
    int tag = lhs_tags[i];
    if (find(wtags.begin(), wtags.end(), tag) != wtags.end())
      tagOrder.push_back(tag);
  }
}

APPSPACK::Citizen::State APPSPACK::Citizen::LHS::getState()
{
  return APPSPACK::Citizen::StopRequest;
}

const string& APPSPACK::Citizen::LHS::getName() const
{
  return name;
}

bool APPSPACK::Citizen::LHS::worker()
{
  cerr << "Worker should be not be called, LHS only requires one citizen." << endl;
  throw "APPSPACK Error";
}

string APPSPACK::Citizen::LHS::IntToString(int n) const
{
  ostringstream ss;
  ss << n;
  return ss.str();
}

void APPSPACK::Citizen::LHS::StringToVector(const string& item, 
					    APPSPACK::Vector& x) const
{
  x.resize(0);
  istringstream ss(item);
  
  double xi;
  while (ss >> xi)
    x.push_back(xi);
}

void APPSPACK::Citizen::LHS::readOutputFile(APPSPACK::Matrix& TP)
{
  TP.clear();

  ifstream fin;
  string name = samplesFile + "." + fileTag;
  fin.open(name.c_str());
  if (!fin) 
  {
    cerr << "\nWarning - cannot open candiate file \"" << samplesFile << "\"." << endl;
    fin.close();
    return;
  }

  // TP are trial points to pass back.
  Vector xi;
  string line;
  while (getline(fin, line))
  {
    StringToVector(line, xi);
    TP.addRow(xi);
  }  
  fin.close();
  unlink(name.c_str());
}

void APPSPACK::Citizen::LHS::addTrialPoints(ConveyorList& Wnew, const Matrix& TP)
{
  // Set up remaining empty variables to create a parentless point.
  Vector f;
  string msg="(" + name + ")";
  
  for (int i=0; i<TP.getNrows(); i++)
  {
    const Vector& x=TP.getRow(i);
    Point* p = new Point(x, f, defaultStep, 0.0, combiner, msg);    
    Wnew.push(p);
  }
}

