// $Id: APPSPACK_Citizen_GSS.cpp,v 1.3 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_GSS.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_GSS.cpp
  \brief Implements APPSPACK::Citizen_GSS
*/

#include "APPSPACK_Citizen_GSS.hpp"

APPSPACK::Citizen::GSS::GSS(Parameter::List& params_in,
			    const Constraints::Linear& constraints_in,
			    Combiner::Generic& combiner_in,
			    const string name_in) :
  params(params_in.sublist(name_in)),
  gss_iterator(params, constraints_in, combiner_in),
  name(name_in+"(GSS)"),
  constraints(constraints_in),
  maxQueue(params.getParameter("Max Queue Size", 0)),
  initialTag(-1),
  debug(params.getParameter("Mediator Debug", 0))
{
}

APPSPACK::Citizen::GSS::~GSS()
{
  exchangeList.prune();
}

void APPSPACK::Citizen::GSS::preprocess()
{
  if (Print::doPrint(Print::InitialData))
  {      
    cout << "\n";
    cout << "###########################################" << "\n";
    cout << "###   APPSPACK Initialization Results   ###" << endl;
    cout << "\n";
    
    gss_iterator.printInitializationInformation();
        
    cout << "\n";
    cout << "### End APPSPACK Initialization Results ###" << endl;
    cout << "###########################################" << "\n";
  }
  
  if (Print::doPrint(Print::NewBestPoint))	
    gss_iterator.printBestPoint("New Min");
  if (Print::doPrint(Print::NewBestDirections))
    gss_iterator.printDirections("New Directions");
  initialTag=gss_iterator.getBestTag();
}

void APPSPACK::Citizen::GSS::postprocess()
{
  // Print the final solution
  if (Print::doPrint(Print::FinalSolution))
  {
    cout << "\nFinal State: " << gss_iterator.getState() << "\n";
    gss_iterator.printBestPoint("Final Min");
  }
  
  // Print solution file if specified by user.
  string solfile;
  solfile = params.getParameter("Solution File", solfile);
  if (!solfile.empty())
    gss_iterator.writeSolutionFile(solfile);  
}

void APPSPACK::Citizen::GSS::exchange(const APPSPACK::ConveyorList& R,
				      APPSPACK::ConveyorList& Wnew, 
                                      const map<string, vector<int> >& ownermap)
{
  const vector<int>& citizen_tags = ownermap.find(name)->second;
  exchangeList.copyFrom(R, citizen_tags);
  printPreDiagnostics();
  popInfeasibleBest();

  // Remove evaluated tags from queueTags list.
  vector<int> rtags;
  exchangeList.getTagList(rtags);
  for (int i=0; i<rtags.size(); i++)
  {
    vector<int>::iterator it = find(queueTags.begin(), queueTags.end(), rtags[i]);    
    if (it != queueTags.end())
      queueTags.erase(it);
  }

  // Perform a single APPSPACK GSS iteration.
  if (debug > 3)
  {
    exchangeList.print("GSS Solver received following points");
    cout << "****\n";
  }

  bool newBestFound = gss_iterator.pointExchange(exchangeList);
  if (newBestFound)
  {
    // New best point has been found.  In this case want to prune
    // according to parameter maxQueue before inserting new tags.
    while ( (queueTags.size() > maxQueue) )
      queueTags.pop_back();
  }

  // exchangeList now holds new list of points to add to the queue.
  vector<int> newTags;
  exchangeList.getTagList(newTags);
  queueTags.insert(queueTags.end(), newTags.begin(), newTags.end());

  // Tags for new queue list recorded.  
  // Add points to external queue list and prune.
  exchangeList.copyTo(Wnew);

  printPostDiagnostics(newBestFound);

  exchangeList.prune();
}

void APPSPACK::Citizen::GSS::popInfeasibleBest()
{
  while (!exchangeList.isEmpty())
  {
    if (constraints.isFeasible(exchangeList.best().getX()))
      break;
    
    cout << "Citizen " << name << ": Popping off best but infeasible point." << endl;
    Point* ptr = exchangeList.popBest();
    delete ptr;
  }
}

void APPSPACK::Citizen::GSS::vote(const APPSPACK::ConveyorList& W,
				  const map<string, vector<int> >& ownermap,
				  vector<int>& tagOrder)
{
  tagOrder.clear();

  // Get list of tags in the queue W.
  vector<int> wtags;
  W.getTagList(wtags);
  
  // Never delete initial point if queued.
  if (find(wtags.begin(), wtags.end(), initialTag) != wtags.end())
    tagOrder.push_back(initialTag);
  
  // We now loop through the current list of queueTags (queueTags might contain pending
  // evaluations) so we will have to be careful not to vote on tags not in W.
  for (int i=0; i<queueTags.size(); i++)
  {
    int tag = queueTags[i];
    if (find(wtags.begin(), wtags.end(), tag) != wtags.end())
      tagOrder.push_back(tag);
  }

}

APPSPACK::Citizen::State APPSPACK::Citizen::GSS::getState()
{
  if (gss_iterator.getState() == APPSPACK::Iterator::Continue)
    return APPSPACK::Citizen::PassiveContinue;
  return APPSPACK::Citizen::StopRequest;
}

const string& APPSPACK::Citizen::GSS::getName() const
{
  return name;
}

void APPSPACK::Citizen::GSS::getBestX(APPSPACK::Vector& x)
{
  x = gss_iterator.getBestX();
}

void APPSPACK::Citizen::GSS::getBestVecF(APPSPACK::Vector& f)
{
  f = gss_iterator.getBestVecF();
}

// PRIVATE
void APPSPACK::Citizen::GSS::printPreDiagnostics() const
{
  if (Print::doPrint(Print::EvaluatedPoints))
    exchangeList.print("Trial Points After Conveyor");
}

// PRIVATE
void APPSPACK::Citizen::GSS::printPostDiagnostics(bool newBestFound) const
{
  if (Print::doPrint(Print::Directions))
    gss_iterator.printDirections("Directions After Trial Point Generation");
  
  if (Print::doPrint(Print::UnevaluatedPoints))
    exchangeList.print("Trial Points Before Conveyor");  

  if (newBestFound)
  {
    if (Print::doPrint(Print::NewBestPoint))	
      gss_iterator.printBestPoint("New Min");
    if (Print::doPrint(Print::NewBestDirections))
      gss_iterator.printDirections("New Directions");
  }
}

