//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file NAPPSPACK_Method_Lagrange.hpp
  \brief Class description for NAPPSPACK::Method::Lagrange
*/

#ifndef NAPPSPACK_METHOD_LAGRANGE
#define NAPPSPACK_METHOD_LAGRANGE

#include "APPSPACK_Common.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Combiner_Lagrange.hpp"
#include "NAPPSPACK_Method_Generic.hpp"

namespace NAPPSPACK
{
namespace Method
{

/*! \brief A class implementing an Augmented Lagrangian approach for handling nonlinear constraint.

This class uses an instance of the class APPSPACK::Combiner::Lagrange to set up a sequence
of linearly consrained subproblem that minimize the Augmented Lagrangian subject to linear
constraints. 
*/
class Lagrange : public Generic
{
public:
  //! Constructor 
  Lagrange(APPSPACK::Parameter::List& params_in);

  //! Derived.
  virtual void update(APPSPACK::Vector& x, APPSPACK::Vector& fc,
		      const APPSPACK::Solver::State& state);
  
  //! Derived.
  virtual bool isOptimal(const APPSPACK::Vector& fc) const;

  //! Derived.
  virtual void printMajor(const APPSPACK::Vector& x, const APPSPACK::Vector& fc) const;

  //! Derived.
  virtual APPSPACK::Combiner::Lagrange& getCombiner();

  //! Derived.
  virtual const string& getName();

  //! Derived.
  virtual double infnorm(const APPSPACK::Vector& fc) const;

private:

  //! Return value of theta based upon lambda and mu value stored in combiner
  double theta() const;
  //! Updates parameters corresponding to a suit of major iterations
  void updateAPPSPACKParams(double tol);
  //! Hestenes-Powell update
  void updateLambda(const APPSPACK::Vector& fc);

private:
  //! Augmented Lagrangian combiner. Used to evaluate meritn function with respect to nonlinear constraints
  APPSPACK::Combiner::Lagrange combiner;
  //! Stores parameter list to be passed to APPSPACK for each subproblem solve
  APPSPACK::Parameter::List& params;

  //@{ Augmented Lagrangian algorithm parameters

  //! Maximum number of subproblems solved
  int maxit;
  
  //! If constraint violation is less than this parameter, we update lambda
  double etak;
  //! Constant used in updating etak when min(gamma_s, max(mu)) does not decrease
  double beta_eta;
  //! Constant used in updating etak when min(gamma_s, max(mu)) decreases
  double alpha_eta;
  //! Constant used in updating etak when min(gamma_s, max(mu)) decreases
  double eta_s;

  //! Determines fraction by which mu is reduced
  double tau;

  //! Determines accuracy of subproblem solve up to a scale factor
  double wk;
  //! Constant used to update wk when min(gamma_s, max(mu)) decreases
  double w_s;
  //! Constant used to update wk when min(gamma_s, max(mu)) decreases
  double alpha_w;
  //! Constant used to update wk when min(gamma_s, max(mu)) does not decrease
  double beta_w;

  //! min(gamma_s, max(mu)).  Current mu is a scalar, but may be made vector later
  double alphak;
  //! Places upper bound on alphak
  double gamma_s;
  
  //! Used in determining theta.  Larger values imply looser subproblem solve tolerances
  double theta_tol;
  //! Step tolerance given to APPSPACK for a given subproblem
  double Deltak;
  //! Constraint violation must be less than this amount prior to exiting
  double etastar;
  //! Deltak must be less than this value prior to exiting
  double Deltastar;
  //@}

  //! Used to test features of algorithm
  //  NAPPSPACK::Tester tester;
  //! Name of method.
  const string methodName;
};

}
}
#endif
