#!/usr/bin/perl -w
#################################################################
#
# Name:	wh.pl
#
# Description:
#   Use this look for files on your path using regular expressions.
#
# ChangeLog:
#          1994   tpg   Initial coding
#   11 Jun 1998   tpg   Initial serious coding
#   22 Oct 2003   tpg   Improve formatting in case someone sees this
#   21 Aug 2004   tpg   Finally details for public release
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation; See http://www.gnu.org/copyleft/gpl.html
################################################################
use strict;

use File::Basename;
use Getopt::Long;
my($me, $mepath, $mesuffix) = fileparse($0, '\.pl');
(my $version = '$Revision: 1855 $ ') =~ tr/[0-9].//cd;

#  These are extensions for DOS filesystems
my @dosext = ('exe', 'com', 'bat', 'dll', 'pl');
#  Always add these paths to @PATH, just in case
my @add2path = ('/sbin', '/usr/sbin');

#--------------------------------------------------------------
#   Initialization
#--------------------------------------------------------------
my %opts = ();
Getopt::Long::GetOptions( \%opts,qw(
    debug help exactly endswith startswith
    )) || die "Failed to parse options\n";

#   Simple help if requested
if ($opts{help} || $#ARGV < 0) {
    print STDERR "$me [-options] pgm1 [pgm2 .. pgmN]\n" .
        "Revision: $version\n" .
        "Show where on your PATH a program can be found.\n" .
        "The following directories are search in addition to your PATH" .
        "\n  " . join("\n  ", @add2path) . "\n" .
        "More details available by entering: perldoc $0\n\n";
    if ($opts{help}) { system("perldoc $0"); }
    exit 1;
}

#  Figure out if this is DOS or Unix. Get separator for paths
my $os = 'unix';
my $pathchar = ':';
if (exists($ENV{'OS'}) && $ENV{'OS'} =~ /win/i) {
    $os = 'win';
    $pathchar = ';';     # PATH separator
}
if ($opts{debug}) { print "    Operating System=$os\n"; }

#--------------------------------------------------------------
#   Get an array of all the directory paths in the PATH env. variable
#   Get the ordered, unique list of directories
#--------------------------------------------------------------
my %h = ();
my @p = split(/$pathchar/,$ENV{PATH});
foreach (@p) { $h{$_} = 1; }
my @path = ();
foreach (@p, @add2path) { if ($h{$_}) { push @path,$_; delete($h{$_}); } }

my $found;
foreach my $pgm (@ARGV) {
    #    Pgm does not have a partial path
    #    Unless directed not to, add * to start,end of pgm
    my $rx = $pgm;
    if ($opts{startswith}) { $rx = '^' . $pgm; }
    elsif ($opts{endswith}) { $rx = $pgm . '$'; }
    elsif ($opts{exactly}) { $rx = '^' . $pgm . '$'; }
    $found = 0;
    foreach my $dir (@path) {
        if ($dir eq '') { next; }
        if (! -e $dir) {       # Directory does not exist
            if ($opts{debug}) { print "    PATH directory '$dir' does not exist\n"; }
            next;
        }
        if ($opts{debug}) { print "    Checking path '$dir'\n"; }
        my $res = GetExecutable($dir, $rx);
        if ($res) { print $res; $found++; }
    }
    if (! $found) { print "'$pgm' not found in PATH\n"; }
}

exit 0;

#==================================================================
# Subroutine:
#   GetExecutable
#
# Description:
#   Returns path to an executable file 
#
# Arguments:
#   dir - directory of file
#   rxpgm - rx of pgm
#
# Return:
#   'none' or path to a file
#
#==================================================================
sub GetExecutable {
    my($dir, $rxpgm) = @_;

    #  Open the directory and get list of all files
    if (! opendir(THISDIR, $dir)) { return undef(); }
    my @files = readdir THISDIR;
    closedir(THISDIR);

    #  For each file in directory, see if it matches
    #  the name of we are looking for
    my @names = ();
    foreach my $f2 (@files) {
        if ($f2 !~ /$rxpgm/) { next; }
        if ($os eq 'win') {         # Windows specific code
            #   Check the file extension for DOS
            foreach my $e (@dosext) {
                if ($f2 =~ /\.${e}$/) { push @names,"$dir\\$f2"; }
            }
        }
        else {                      # Unix specific
            #   If the file is executable and not a directory...
            #   (note: this is more accurate in AFS)
            my ($dev, $ino, $mode) = stat("$dir/$f2");
            if (defined($mode) & 0111 && !($mode & 040000)) {
                push @names,"$dir/$f2";
            }
        }
    }
    if (! @names) { return undef(); }
    return join("\n",@names) . "\n";
}


#==================================================================
#   Perldoc Documentation
#==================================================================
__END__

=head1 NAME

wh.pl - a smarter 'which'

=head1 SYNOPSIS

  	wh.pl ls rm             # Looks for /.*ls.*/ and /.*rm.*/

  	wh.pl 'l[fc]s'          # Looks for /l[fc]s/

  	wh.pl -exactly ls       # Looks for /^ls$/

  	wh.pl -startswith ls    # Looks for /^ls/

  	wh.pl -endswith ls      # Looks for /ls$/


=head1 DESCRIPTION

This program can be used to show executables on your PATH.
Sometimes you remember PART of the name of an executable.
You can use this program to show you all executables on your
PATH with the executable name partially specified.

This program should work on either Unix or DOS systems where Perl is installed.

Use Perl regular expressions with a program name to find
executables on the path. Don't forget, you may need to escape
special characters so the shell will not expand them.
E.g. wh.pl 'l[fc]s'

On Unix systems in addition to your normal PATH, the program will
search in /sbin and /usr/sbin to find executables that might be
on the system admin PATH, but not your PATH.

The default action is to show any executable whose name contains
the string you provide.
For instance, 'wh.pl ls' would find all programs which have the
characters 'ls' in the name (quite a long list typically).
This would include 'ls', 'lsmod', and 'filelsize' as examples.

On DOS systems, the program will look for executables with these
extensions: 'exe', 'com', 'bat', 'dll', and 'pl'.
On a DOS system, the command 'wh.pl ls' might find 'ls.exe' or 'ls.bat'
among other possibilities.

=head2 OPTIONS

=over 4

=item B<-debug>

Enables debugging to show you where it is searching.
Additionally it will also tell you about directories that
are missing in your PATH.

=item B<-endswith>

Indicates the the program name 'ends with' the name provided.
Specifying '-endswith ls' would find '/bin/fills', but
not '/bin/ls'.

=item B<-exactly>

Do not use any regular expression matching when looking
for the program.

=item B<-help>

Shows you this help file.

=item B<-startswith>

Indicates the the program name 'starts with' the name provided.
Specifying '-startswith ls' would find '/bin/ls' and '/bin/lsmod',
but not '/bin/filsize'.

=back

=head2 PARAMETERS

=over 4

=item B<str1 [str2 ... strN]>

This is the string used to find programs in your PATH.
You may specify more than one string (program name) to search for.

=back

=head1 EXIT

If no fatal errors are detected, the program exits with a
return code of 0.

=head1 AUTHOR

Written by Terry Gliedt I<E<lt>tpg@hps.comE<gt>> in 1994-2003.
This is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation; See http://www.gnu.org/copyleft/gpl.html

=cut
