/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file StdOptSolver.h
 *
 * Defines the colin::StdOptSolver class.
 */

#ifndef colin_StdOptSolver_h
#define colin_StdOptSolver_h

#include <acro_config.h>
#include <utilib/_math.h>
#include <utilib/PM_LCG.h>
#include <colin/OptSolver.h>
#include <colin/ColinUtilib.h>

namespace colin {

using utilib::ParameterSet;

/** The \c StdOptSolver class defines an extension of the core COLIN solver
  * class that exploits capability in UTILIB to define a core Solver
  * infrastructure that has proven useful in the development of Coliny.
  */
template <class DomainT, class ResponseT=AppResponse<> >
class StdOptSolver : public OptSolver<DomainT,ResponseT>
{
public:

  ///
  StdOptSolver();

  ///
  ~StdOptSolver() {}

  /** Perform minimization.
    * This method is defined by the subclass.  The \c check_convergence
    * method can be used to utilize the termination controls defined
    * by this class.  Also, the \c debug_io method can be used
    * to print general debugging information in a standard format.  If
    * the developer wishes to provide additional debugging information,
    * the \c virt_debug_io method can be redefined.  This method
    * is called from \c debug_io.
    */
  void minimize();

  ///
  void reset()
		{
		OptSolver<DomainT,ResponseT>::reset();
		prev_iter=0;
		prev_neval=0;
		prev_time=0.0;
		prev_best=-999.999;
		prev_cbest=-999.999;
		prev_print=-1;
		end_iter_flag=false;
		this->curr_iter=0;
		if (seed > 0)
		   pm_rng.set_seed(static_cast<unsigned>(seed));
		}

  ///
  void write(std::ostream& os) const;

protected:

  /** Print debugging information and call \c virt_debug_io for 
    * subclass debugging information.
    */
  void debug_io(std::ostream& os, bool finishing=false);

  /// A single iteration of minimization
  virtual void minimize_iteration() {}

  /// If true, then print all debugging info
  bool Debug_all;

  /// If true, then print time info
  bool Debug_time;

  /// If true, then print the number of evaluations
  bool Debug_neval;

  /// If true, then print the iteration info
  bool Debug_iter;

  /// If true, then print the value of the best point
  bool Debug_best;

  /// If true, then print info about the best point
  bool Debug_best_point;

  /// If true, then print info about the optimizer
  bool Debug_opt_stats;

  /// The previous iteration that had a different value
  int prev_iter;

  /// The number of evaluations at iteration \c prev_iter
  int prev_neval;

  /// The time at iteration \c prev_iter
  double prev_time;

  /// The value of the best point at iteration \c prev_iter
  double prev_best;

  /// The value of the constraint violation at iteration \c prev_iter
  double prev_cbest;

  /// TODO
  bool end_iter_flag;

  ///
  bool print_final_points;

  /// TODO
  int prev_print;

  /// Random number generator
  utilib::PM_LCG pm_rng;

  /// Seed for the random number generator
  int seed;
};


//============================================================================
//
//
template <class DomainT, class ResponseT>
StdOptSolver<DomainT,ResponseT>
	::StdOptSolver()
{
Debug_all = false;
Debug_time=false;
Debug_neval=false;
Debug_iter=false;
Debug_best=false;
Debug_best_point=false;
Debug_opt_stats=false;
print_final_points=false;

prev_iter=0;
prev_neval=0;
prev_time=0.0;
prev_best=-999.999;
prev_cbest=-999.999;
end_iter_flag=false;
prev_print=-1;

seed=0;
OptSolver<DomainT,ResponseT>::rng = &pm_rng;

/// Debugging parameters

ParameterSet::create_categorized_parameter("seed",seed,
        "<int>","0","The seed value for the random number generator.  A value\n\tof zero indicates that the current time is used to seed the random\n\tnumber generator",
	"General",
        utilib::ParameterLowerBound<int>(0));

ParameterSet::create_categorized_parameter("debug_time",Debug_time,
	"<bool>","false",
	"Print info about the amount of elapsed time",
	"Debugging");

ParameterSet::create_categorized_parameter("debug_num_evaluations",Debug_neval,
	"<bool>","false",
	"Print info about the number of fevals",
	"Debugging");
ParameterSet::alias_parameter("debug_num_evaluations","debug_neval");

ParameterSet::create_categorized_parameter("debug_iteration_info",Debug_iter,
	"<bool>","false",
	"Print info about the current iteration",
	"Debugging");
ParameterSet::alias_parameter("debug_iteration_info","debug_iter");

ParameterSet::create_categorized_parameter("debug_best",Debug_best,
	"<bool>","false",
	"Print info about the value of the best point found so far",
	"Debugging");

ParameterSet::create_categorized_parameter("debug_best_point",Debug_best_point,
	"<bool>","false",
	"Print info about the best point found so far",
	"Debugging");

ParameterSet::create_categorized_parameter("debug_opt_stats",Debug_opt_stats,
	"<bool>","false",
	"Print general optimization statistics",
	"Debugging");

ParameterSet::create_categorized_parameter("debug_all",Debug_all,
	"<bool>","false",
	"Enable all debug_* options",
	"Debugging");

ParameterSet::create_categorized_parameter("print_final_points",print_final_points,
	"<bool>","false",
	"Print final points",
	"Debugging");
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void StdOptSolver<DomainT,ResponseT>
        ::minimize()
{
this->opt_init();

unsigned int num_iters;
if (this->max_iters <= 0)
   num_iters = MAXINT;
else
   num_iters = this->curr_iter + this->max_iters;

debug_io(ucout);
for (this->curr_iter++; this->curr_iter <= num_iters; this->curr_iter++) {

  if (this->check_convergence(this->best().value()))
     break;

  minimize_iteration();

  debug_io(ucout);
  }

debug_io(ucout,true);
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void StdOptSolver<DomainT,ResponseT>
	::write(std::ostream& os) const
{
OptSolver<DomainT,ResponseT> ::write(os);

os << "Debug_time       " << Debug_time << std::endl;
os << "Debug_neval      " << Debug_neval << std::endl;
os << "Debug_iter       " << Debug_iter << std::endl;
os << "Debug_best       " << Debug_best << std::endl;
os << "Debug_best_point " << Debug_best_point << std::endl;
os << "Debug_opt_stats  " << Debug_opt_stats<< std::endl;
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void StdOptSolver<DomainT,ResponseT>
        ::debug_io(std::ostream& os, bool finishing)
{
//
// Perform a preliminary check to see if anything is going to be
// printed.  This doesn't check for the virt_debug_io routines,
// which are restricted to being printed by Output_freq
//
if ( (this->Output_freq == 0) ||
     ((this->debug == 0) && (this->output_level == 0)) ||
     (this->output_final && !finishing) ) return;

if ((this->Output_freq > 0) &&
     !(((this->curr_iter % this->Output_freq) == 0) ^ finishing)) {
   os << ")\n";
   if (this->best().termination_info != "") {
      os << "[ Termination: ";
      os << this->best().termination_info;
      os << " ]\n";
      }
   return;
   }

this->time_curr = CPUSeconds();

if (this->output_dynamic &&
    ((!finishing && (prev_best == this->best().value())) ||
                    (finishing && (prev_print == prev_iter)))
   ) {
   prev_time  = this->time_curr - this->time_start;
   prev_neval = this->neval();
   prev_best  = this->best().value();
   prev_cbest = this->best().constraint_violation;
   prev_iter  = this->curr_iter;
   return;
   }

if (end_iter_flag && (this->output_level > 0))
   os << ")\n";
else
   end_iter_flag = 1;

if (!finishing && this->output_dynamic &&
    (this->output_level==1) && (this->curr_iter > 0) && (prev_print != prev_iter)) {
   os << "(---COLIN--- Begin Optimizer Iteration -----------------------------------------" << std::endl;
   os << "[\nSummary:\tOpt: " << this->opt_name << "  Iter: " << prev_iter <<
                "  Value: " << prev_best << "  CValue: " << prev_cbest << " Neval: " << prev_neval << "\n]\n";
   os << ")\n";
   }

if (this->output_level > 0)
   os << "(---COLIN--- Begin Optimizer Iteration -----------------------------------------" << std::endl;


//
// Go through list of debuging stats
//
if (((this->Output_freq > 0) &&
     (((this->curr_iter % this->Output_freq) == 0) ^ finishing)) ||
    this->output_dynamic || this->output_final) {

   //
   // User Output
   //
   switch (this->output_level) {
     case 0:            // No user output
        break;

     case 1:
        if (finishing && this->output_dynamic)
           os << "[\nSummary:\tOpt: " << this->opt_name << "  Iter: " << prev_iter <<
                "  Value: " << prev_best << " CValue: " << prev_cbest << " Neval: " << prev_neval << "\n]\n";
        else {
           double time_diff = this->time_curr - this->time_start;
           os << "[\nSummary:\tOpt: " << this->opt_name << "  Iter: " << this->curr_iter <<
                "  Value: ";
           os << this->best().value();
           os << "  CValue: " << this->best().constraint_violation << " Neval: " << this->neval() << "\n]\n";

           if (this->output_dynamic) {
              prev_time  = time_diff;
              prev_neval = this->neval();
              prev_best  = this->best().value();
              prev_cbest = this->best().constraint_violation;
              prev_iter  = this->curr_iter;
              prev_print = this->curr_iter;
              }
           }
        break;

      case 2:
      case 3:
        double time_diff = this->time_curr - this->time_start;
        if (this->output_level == 2)
           os << "[\nNormal:" << std::endl;
        else
           os << "[\nVerbose:" << std::endl;
        os << "\tOptimizer Type:\t\t\t\t" << this->opt_name << std::endl;
        os << "\tIteration Number:\t\t\t" << this->curr_iter << std::endl;
        os << "\tBest Point - Objective Fn:\t\t";
        os << this->best().response.function_value() << std::endl;
        //os << this->best().value;
        //os << std::endl;
	if (this->problem.numNonlinearConstraints() > 0)
           os << "\tBest Point - COLIN Merit Fn:\t\t" << this->best().response.augmented_function_value() << std::endl;
        os << "\tBest Point - Constraint L2 Norm:\t" << this->best().constraint_violation << std::endl;
        os << "\tTotal # Func Evals:\t\t\t" << this->neval() << std::endl;
        os << "\tTotal Time (CPU+System):\t\t" << time_diff << std::endl;
        this->virt_debug_io(os,finishing,this->output_level);
        os << "]\n";
        break;
      };

   if (this->debug) {
      //
      // Debugging Output
      //
      if (Debug_iter || Debug_all)
         os << "[Iter:\t" << this->curr_iter << "]\n";

      if (Debug_neval || Debug_all)
         os << "[Neval:\t" << this->neval() << "]\n";

      if (Debug_time || Debug_all) {
         double time_diff = this->time_curr - this->time_start;
         os << "[Time:\t" << time_diff << "]\n";
         os << "[Time-Stamp:\t" << ElapsedCPUSeconds() << "]\n";
         }

      if (Debug_best || Debug_opt_stats || Debug_all) {
         os << "[Min:\t";
         os << this->best().value();
         os << "]\n";
         }

      if (Debug_best_point || Debug_opt_stats || Debug_all) {
         os << "[Min-point:\n\t" << this->best().point << "]\n";
         }

      //
      // Call virtual function which may print more, optimizer-specific stats
      //
      this->virt_debug_io(os,finishing,-1);
      }
   }

if (finishing && print_final_points) {
   os << "[Final-Points:\n";
   std::vector<DomainT> points;
   this->get_final_points(points);
   for (unsigned int i=0; i<points.size(); i++) {
     os << "Point " << i << " = " << points[i] << std::endl;
     }
   os << "]\n";
   }

if (this->Output_flush) { os.flush(); ucout << utilib::Flush; }
}

}

#endif
