/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblemStateBase.h
 *
 * Defines the colin::OptProblemStateBase class.
 */

#ifndef colin_OptProblemStateBase_h
#define colin_OptProblemStateBase_h

#include <acro_config.h>
#include <utilib/stl_auxillary.h>
#include <colin/OptDomainTraits.h>
#include <colin/BoundTypeArray.h>

namespace colin {

/// Validate that the constraint matrix and rhs match the constraint
/// information provided by the user.
template <class VectorT, class MatrixT>
void validate_linear_constraints(const MatrixT& /*mat*/, const VectorT& /*rhs*/,
				int /*num_params*/,
				unsigned int& numLinearIneqConstraints,
				unsigned int numLinearEqConstraints)
{
//
// TODO check that # columns = num_params
// TODO check that # rows    = rhs size
// TODO check that # norw    < numLinearEqConstraints
//
numLinearIneqConstraints = numLinearEqConstraints = 0;

#if defined(COUGAR) || defined(TFLOPS_SERVICE)
///
/// Some dummy code
///
if (numLinearIneqConstraints > 0)
   numLinearEqConstraints++;
#endif
}


//============================================================================
//============================================================================
// Class OptProblemStateBase
//============================================================================
//============================================================================

/** \class OptProblemStateBase
  *
  * This class contains the data that defines the state of an optimization
  * problem.  
  */
class OptProblemStateBase
{
public:

  /// Constructor
  OptProblemStateBase();

  /// Destructor
  virtual ~OptProblemStateBase() {}

  /// Reset the state
  virtual void reset();

  /**@name Evaluation Data */
  //@{
  /** If true, then use a database of evaluations to check whether each 
    * evaluation has alread been computed.
    */
  bool db_check;

  /// The ID of the last evaluation
  int last_id;

  /// The number of functions
  int numFunctions;

  /// Initialize the active set vector
  void init_request_asv(int mode);

  /// The mode of the request
  int request_mode;

  /// The ASV vector for the requested evaluations
  std::vector<int> REQUEST_ASV;

  /// The ASV vector for the requested function evaluations
  std::vector<int> F_ASV;

  /// The ASV vector for the requested constraint evaluations
  std::vector<int> CF_ASV;

  /// The ASV vector for the requested gradient evaluations
  std::vector<int> G_ASV;

  /// The ASV vector for the requested function + gradient evaluations
  std::vector<int> FG_ASV;

  /// The ASV vector for the requested constraint + constraint gradient evaluations
  std::vector<int> CG_ASV;
  //@}

  /**@name Constraint Information */
  //@{
  /// Setup the constraint counters
  void set_constraint_ctrs(unsigned int lineareq, unsigned int linearineq,
                        unsigned int nonlineareq, unsigned int nonlinearineq)
  	{
	numLinearEqConstraints=lineareq;
	numLinearIneqConstraints=linearineq;
	numNonlinearEqConstraints=nonlineareq;
	numNonlinearIneqConstraints=nonlinearineq;
	numConstraints=numLinearEqConstraints+numLinearIneqConstraints+
               numNonlinearEqConstraints+numNonlinearIneqConstraints;
        //
        // Only reset these if we have to, since the user
        // may have already initialized this data.
        //
	if (constraint_lower_bounds.size() != numConstraints) {
	   constraint_lower_bounds.resize(numConstraints);
	   constraint_lower_bounds << real::negative_infinity;
	   constraint_upper_bounds.resize(numConstraints);
	   constraint_upper_bounds << real::positive_infinity;
           }
	}

  /// Setup the constraint bounds
  void set_constraint_bounds(const std::vector<real>& lower_,
		  				const std::vector<real>& upper_);
  
  /// Get the constraint bounds
  void get_constraint_bounds(std::vector<real>& lower_, std::vector<real>& upper_);
  
  /// The number of linear equality constraints.
  unsigned int numLinearEqConstraints;

  /// The number of linear inequality constraints.
  unsigned int numLinearIneqConstraints;

  /// The number of nonlinear equality constraints.
  unsigned int numNonlinearEqConstraints;

  /// The number of nonlinear inequality constraints.
  unsigned int numNonlinearIneqConstraints;

  /// The total number of constraints (except for bound constraints).
  unsigned int numConstraints;

  //
  /// Lower bounds array 
  std::vector<real> constraint_lower_bounds;
 
  /// Upper bounds array 
  std::vector<real> constraint_upper_bounds;
  //@}

  /**@name Variable Boundary Constraints */
  //@{
  /// Set boundary constraints - dense format
  void set_int_bounds(const std::vector<int>& lower, const std::vector<int>& upper);

  /// Set boundary constraints - dense format
  void set_real_bounds(const std::vector<real>& lower, const std::vector<real>& upper);
 
  /// Set boundary constraints - sparse format
  virtual void set_bounds(const char* format) = 0;
 
  /// Get boundary constraints - dense format
  void get_int_bounds(std::vector<int>& lower, std::vector<int>& upper) const;
 
  /// Get boundary constraints - dense format
  void get_real_bounds(std::vector<real>& lower, std::vector<real>& upper) const;

  /// Lower bounds array 
  std::vector<real> real_lower_bounds;
 
  /// Upper bounds array 
  std::vector<real> real_upper_bounds;

  /// Lower bounds array 
  std::vector<int> int_lower_bounds;
 
  /// Upper bounds array 
  std::vector<int> int_upper_bounds;

  /// Returns true if all search dimensions are bounded
  bool has_all_bounds()
	{
	if (!enforcing_bound_constraints) return false;
        for (size_type i=0; i<num_real_params; i++) {
          if ((real_lower_bound_type[i] == no_bound) ||
	      (real_upper_bound_type[i] == no_bound))
	     return false;
          }
        for (size_type i=0; i<num_int_params; i++) {
          if ((int_lower_bound_type[i] == no_bound) ||
	      (int_upper_bound_type[i] == no_bound))
	     return false;
          }
        return true;
	}

  /// true if enforcing bound constraints
  bool enforcing_bound_constraints;

  /// The type of real lower bound
  BoundTypeArray real_lower_bound_type;

  /// The type of real upper bound
  BoundTypeArray real_upper_bound_type;

  /// The type of integer lower bound
  BoundTypeArray int_lower_bound_type;

  /// The type of integer upper bound
  BoundTypeArray int_upper_bound_type;

  //@}

  /// The number of real parameters
  unsigned int num_real_params;

  /// The number of integer parameters
  unsigned int num_int_params;

  /// The number of boolean parameters
  unsigned int num_binary_params;

  /// True if using gradients
  bool using_gradients;

  /// True if using hessians
  bool using_hessians;

  /// Returns true if the search domain includes reals
  virtual bool domain_reals() = 0;

  /// Returns true if the search domain includes integers
  virtual bool domain_integers() = 0;

  /// Returns true if the search domain includes booleans
  virtual bool domain_binarys() = 0;

};


} // namespace colin

#endif
