/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file ColinUtilib.h
 *
 * Defines items that are needed to interface with UTILIB data structures
 * within COLIN.
 */

#if !defined(DOXYGEN)
#ifndef colin_ColinUtilib_h
#define colin_ColinUtilib_h

#include <acro_config.h>
#include <colin/real.h>
#include <utilib/stl_auxillary.h>
#include <utilib/NumArray.h>
#include <utilib/MixedIntVars.h>
#include <colin/OptDomainTraits.h>
#include <colin/AppResponse.h>


//============================================================================
//
// ResponseTypes object using UTILIB arrays
//
namespace colin {

#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
typedef AppResponseTypes<utilib::BasicArray> AppResponseTypes_Utilib;
#else
typedef AppResponseTypes<utilib::BasicArray<real>,
		        utilib::BasicArray<real>,
		        utilib::BasicArray<int> > AppResponseTypes_Utilib;
#endif
typedef AppResponse<AppResponseTypes_Utilib> AppResponse_Utilib;

}

//============================================================================
//
// DomainTrait definitions using UTILIB arrays
//
SetOptDomainTrait(utilib::BasicArray<double>,reals,true)
SetOptDomainTrait(utilib::BasicArray<double>,linearconstraints,true)
SetOptDomainTrait(utilib::BasicArray<double>,boundconstraints,true)
SetOptDomainTrait(utilib::BasicArray<int>,integers,true)
SetOptDomainTrait(utilib::BasicArray<int>,linearconstraints,true)
SetOptDomainTrait(utilib::BasicArray<int>,boundconstraints,true)

SetOptDomainTrait(utilib::NumArray<double>,reals,true)
SetOptDomainTrait(utilib::NumArray<double>,linearconstraints,true)
SetOptDomainTrait(utilib::NumArray<double>,boundconstraints,true)
SetOptDomainTrait(utilib::NumArray<int>,integers,true)
SetOptDomainTrait(utilib::NumArray<int>,linearconstraints,true)
SetOptDomainTrait(utilib::NumArray<int>,boundconstraints,true)

SetOptDomainTrait(utilib::pvector<double>,reals,true)
SetOptDomainTrait(utilib::pvector<double>,linearconstraints,true)
SetOptDomainTrait(utilib::pvector<double>,boundconstraints,true)
SetOptDomainTrait(utilib::pvector<int>,integers,true)
SetOptDomainTrait(utilib::pvector<int>,linearconstraints,true)
SetOptDomainTrait(utilib::pvector<int>,boundconstraints,true)


//============================================================================
//
// DomainTrait definitions using UTILIB MixedIntVars object
//
SetOptDomainTrait(utilib::MixedIntVars,reals,true)
SetOptDomainTrait(utilib::MixedIntVars,integers,true)
SetOptDomainTrait(utilib::MixedIntVars,linearconstraints,true)
SetOptDomainTrait(utilib::MixedIntVars,boundconstraints,true)

namespace colin {

template <>
inline void map_domain<std::vector<int>, utilib::BasicArray<int> >(
		std::vector<int>& target_domain,
		const utilib::BasicArray<int>& input_params)
{
target_domain.resize(input_params.size());
target_domain << input_params;
}

template <>
inline void map_domain<utilib::BasicArray<int>, std::vector<int> >(
		utilib::BasicArray<int>& target_domain,
		const std::vector<int>& input_params)
{
target_domain.resize(input_params.size()); 
target_domain << input_params;
}



template <>
inline void map_domain<std::vector<double>, utilib::BasicArray<double> >
			(std::vector<double>& target_domain,
			const utilib::BasicArray<double>& input_params)
{ target_domain.resize(input_params.size()); target_domain << input_params; }

template <>
inline void map_domain<utilib::BasicArray<double>, std::vector<double> >
			(utilib::BasicArray<double>& target_domain,
			const std::vector<double>& input_params)
{ target_domain.resize(input_params.size()); target_domain << input_params; }



template <>
inline void map_domain<utilib::pvector<int>, utilib::BasicArray<int> >(
		utilib::pvector<int>& target_domain,
		const utilib::BasicArray<int>& input_params)
{
target_domain.resize(input_params.size());
target_domain << input_params;
}

template <>
inline void map_domain<utilib::BasicArray<int>, utilib::pvector<int> >(
		utilib::BasicArray<int>& target_domain,
		const utilib::pvector<int>& input_params)
{
target_domain.resize(input_params.size()); 
target_domain << input_params;
}



template <>
inline void map_domain<utilib::pvector<double>, utilib::BasicArray<double> >
			(utilib::pvector<double>& target_domain,
			const utilib::BasicArray<double>& input_params)
{ target_domain.resize(input_params.size()); target_domain << input_params; }

template <>
inline void map_domain<utilib::BasicArray<double>, utilib::pvector<double> >
			(utilib::BasicArray<double>& target_domain,
			const utilib::pvector<double>& input_params)
{ target_domain.resize(input_params.size()); target_domain << input_params; }



template <>
inline void map_domain<utilib::BasicArray<double>, utilib::NumArray<double> >
			(utilib::BasicArray<double>& target_domain,
			const utilib::NumArray<double>& input_params)
{ target_domain.resize(input_params.size()); target_domain << static_cast<utilib::BasicArray<double> >(input_params); }

template <>
inline void map_domain<utilib::NumArray<double>, utilib::BasicArray<double> >
			(utilib::NumArray<double>& target_domain,
			const utilib::BasicArray<double>& input_params)
{ target_domain.resize(input_params.size()); target_domain << input_params; }



template <>
inline void map_domain<utilib::pvector<double>, utilib::NumArray<double> >
			(utilib::pvector<double>& target_domain,
			const utilib::NumArray<double>& input_params)
{ target_domain.resize(input_params.size()); target_domain << input_params; }

template <>
inline void map_domain<utilib::NumArray<double>, utilib::pvector<double> >
			(utilib::NumArray<double>& target_domain,
			const utilib::pvector<double>& input_params)
{ target_domain.resize(input_params.size()); target_domain << input_params; }



template <>
inline void map_domain<utilib::BasicArray<double>, utilib::BasicArray<double> >
			(utilib::BasicArray<double>& target_domain,
			const utilib::BasicArray<double>& input_params)
{ target_domain.resize(input_params.size()); target_domain << input_params; }


template <>
inline void map_domain<utilib::pvector<int>, utilib::MixedIntVars >(
		utilib::pvector<int>& target_domain,
		const utilib::MixedIntVars& input_params)
{
target_domain.resize(input_params.numGeneralIntVars());
target_domain << input_params;
}

template <>
inline void map_domain<utilib::MixedIntVars, utilib::pvector<int> >(
		utilib::MixedIntVars& target_domain,
		const utilib::pvector<int>& input_params)
{
target_domain.resize(0,input_params.size(),0); 
target_domain << input_params;
}


template <>
inline void map_domain<std::vector<int>, utilib::MixedIntVars >(
		std::vector<int>& target_domain,
		const utilib::MixedIntVars& input_params)
{
target_domain.resize(input_params.numGeneralIntVars());
target_domain << input_params;
}

template <>
inline void map_domain<utilib::MixedIntVars, std::vector<int> >(
		utilib::MixedIntVars& target_domain,
		const std::vector<int>& input_params)
{
target_domain.resize(0,input_params.size(),0); 
target_domain << input_params;
}


template <>
inline void map_domain<utilib::BasicArray<int>, utilib::MixedIntVars >(
		utilib::BasicArray<int>& target_domain,
		const utilib::MixedIntVars& input_params)
{
target_domain.resize(input_params.numGeneralIntVars());
target_domain << input_params;
}

template <>
inline void map_domain<utilib::MixedIntVars, utilib::BasicArray<int> >(
		utilib::MixedIntVars& target_domain,
		const utilib::BasicArray<int>& input_params)
{
target_domain.resize(0,input_params.size(),0); 
target_domain << input_params;
}


template <>
inline void map_domain<utilib::pvector<double>, utilib::MixedIntVars >(
		utilib::pvector<double>& target_domain,
		const utilib::MixedIntVars& input_params)
{
target_domain.resize(input_params.numDoubleVars());
target_domain << input_params;
}

template <>
inline void map_domain<utilib::MixedIntVars, utilib::pvector<double> >(
		utilib::MixedIntVars& target_domain,
		const utilib::pvector<double>& input_params)
{
target_domain.resize(0,0,input_params.size()); 
target_domain << input_params;
}


template <>
inline void map_domain<std::vector<double>, utilib::MixedIntVars >(
		std::vector<double>& target_domain,
		const utilib::MixedIntVars& input_params)
{
target_domain.resize(input_params.numDoubleVars());
target_domain << input_params;
}

template <>
inline void map_domain<utilib::MixedIntVars, std::vector<double> >(
		utilib::MixedIntVars& target_domain,
		const std::vector<double>& input_params)
{
target_domain.resize(0,0,input_params.size()); 
target_domain << input_params;
}


template <>
inline void map_domain<utilib::BasicArray<double>, utilib::MixedIntVars >(
		utilib::BasicArray<double>& target_domain,
		const utilib::MixedIntVars& input_params)
{
target_domain.resize(input_params.numDoubleVars());
target_domain << input_params;
}

template <>
inline void map_domain<utilib::MixedIntVars, utilib::BasicArray<double> >(
		utilib::MixedIntVars& target_domain,
		const utilib::BasicArray<double>& input_params)
{
target_domain.resize(0,0,input_params.size()); 
target_domain << input_params;
}


} // namespace colin

#endif
#endif
