/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file AnalysisCode.h
 *
 * Defines the colin::AnalysisCode class.
 */

#ifndef colin_AnalysisCode_h
#define colin_AnalysisCode_h

#include <acro_config.h>
#include <utilib/std_headers.h>
#include <utilib/LinkedList.h>
#include <utilib/stl_auxillary.h>
#include <utilib/CommonIO.h>
#include <utilib/exception_mngr.h>
#include <colin/ColinGlobals.h>

namespace colin {

/** \class AnalysisCode 
  *
  * Base class providing common functionality for derived classes
  * \c SysCallAnalysisCode and \c ForkAnalysisCode which spawn separate
  * processes for managing simulations.
  *
  * The AnalysisCode class hierarchy provides simulation spawning
  * services for ApplicationInterface derived classes and alleviates
  * these classes of some of the specifics of simulation code
  * management.  The hierarchy does not employ the letter-envelope
  * technique since the ApplicationInterface derived classes
  * instantiate the appropriate derived AnalysisCode class directly.
  *
  * Output verbosity is observed within AnalysisCode 
  * (file operations verbosity) with the following meanings:
  *   "quiet"  :   quiet iterators,   quiet interface,   quiet file ops
  *   "normal" :   quiet iterators, verbose interface,   quiet file ops
  *   "verbose": verbose iterators,   debug interface, verbose file ops
  *   "debug"  :   debug iterators,   debug interface, verbose file ops
  * where "quiet", "verbose" and "debug" must be user specified and "normal"
  * is the default for no user specification.  Since quiet is the file 
  * operations default, the verboseFlag need only detect an explicit 
  * "verbose" or "debug" request.
  */
template <class DomainT, class ResponseT, class StringT=std::string>
class AnalysisCode
{
public:

  /// Setup the analysis code
  void setup(const StringT& program_name,
                 const StringT& input_filename,
                 const StringT& output_filename,
                 const bool tag_flag=true, const bool save_flag=false,
		 const bool aprepro=false)
		{
		programNames.push_back(program_name);
		numPrograms = 1;
		parametersFileName = input_filename;
		resultsFileName = output_filename;
		fileTagFlag = tag_flag;
		fileSaveFlag = save_flag;
		apreproFlag = aprepro;
		}
  /// Setup the analysis code (with input/output filters)
  void setup(const StringT& ifilter_name, const StringT& ofilter_name,
                 const StringT& program_name,
                 const StringT& input_filename,
                 const StringT& output_filename,
                 const bool tag_flag=true, const bool save_flag=false,
		 const bool aprepro=false)
		{
		iFilterName = ifilter_name;
		oFilterName = ofilter_name;
		setup(program_name, input_filename, output_filename,
				tag_flag, save_flag, aprepro);
		}

  /** Define modified filenames from user input by handling Unix
    * temp file and tagging options.
    *
    * These names are used in writing the parameters file, in defining the 
    * \c CommandShell overloaded operator behavior in \c SyscallApplication, in 
    * reading the results file, and in file cleanup. */
  void define_filenames(const int id, const int num_analysis_servers=1);

  /// write the variables and active set vector objects to the
  /// parameters file in either standard or aprepro format
  void write_parameters_file(const DomainT& vars, 
                               const std::vector<int>& asv, const int id);

  /// read the response object from the results file
  void read_results_file(ResponseT& response, const int id);

#if defined(COUGAR) || defined(TFLOPS_SERVICE)
  /// return programNames
  const utilib::LinkedList<StringT>& program_names() const
#else
  /// return programNames
  const std::list<StringT>& program_names() const
#endif
		{return programNames;}

  /// return iFilterName
  const StringT& input_filter_name() const
		{return iFilterName;}
  /// return oFilterName
  const StringT& output_filter_name() const
		{return oFilterName;}

  /// return modifiedParamsFileName
  const StringT& modified_parameters_filename() const
		{return modifiedParamsFileName;}
  /// return modifiedResultsFileName
  const StringT& modified_results_filename()    const
		{return modifiedResultsFileName;}

  /// return the entry in resultsFNameList corresponding to id
  const StringT& results_fname(const int id) const
		{
		std::map<int,std::string>::const_iterator tmp = resultsFName.find(id);
		if (tmp == resultsFName.end())
		   EXCEPTION_MNGR(std::runtime_error,
			"AnalysisCode::results_fname - unknown id " << id);
  		return tmp->second;
		}

  /// set quietFlag
  void quiet_flag(bool flag)
		{quietFlag = flag;}
  /// return quietFlag
  bool quiet_flag() const
		{return quietFlag;}

protected:

  /// Protected constructor, since only derived classes should be created
  AnalysisCode();

  /// Destructor
  virtual ~AnalysisCode() {}

  /// flag set by master processor to quiet output from slave processors
  bool quietFlag;
  /// flag for additional analysis code output if method verbosity is set
  bool verboseFlag;
  /// flags tagging of parameter/results files
  bool fileTagFlag;
  /// flags retention of parameter/results files
  bool fileSaveFlag;
  /// flags use of the APREPRO (the Sandia "A PRE PROcessor" utility)
  /// format for parameter files
  bool apreproFlag;

  /// the name of the input filter (input_filter user specification)
  StringT iFilterName;
  /// the name of the output filter (output_filter user specification)
  StringT oFilterName;

  /// the names of the analysis code programs (analysis_drivers user
  /// specification)
#if defined(COUGAR) || defined(TFLOPS_SERVICE)
  utilib::LinkedList<std::string> programNames;
#else
  std::list<std::string> programNames;
#endif
  /// the number of analysis code programs (length of programNames list)
  size_t numPrograms;

  /// the name of the parameters file from user specification
  StringT parametersFileName;
  /// the parameters file name actually used (modified with tagging
  /// or temp files)
  StringT modifiedParamsFileName;
  /// the name of the results file from user specification
  StringT resultsFileName;
  /// the results file name actually used (modified with tagging
  /// or temp files)
  StringT modifiedResultsFileName;

  /// list of parameters file names used in spawning function evaluations
  std::map<int,std::string> parametersFName;
  /// list of results file names used in spawning function evaluations
  std::map<int,std::string> resultsFName;

};



//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
AnalysisCode<DomainT,ResponseT,StringT>::AnalysisCode()
  : quietFlag(false)
{
verboseFlag = ColinGlobals::output_level == "debug" ||
                        ColinGlobals::output_level == "verbose";
quietFlag = ColinGlobals::output_level == "quiet";
}


//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void AnalysisCode<DomainT,ResponseT,StringT>
	::define_filenames(const int id, const int analysis_servers)
{
// Different analysis servers _must_ share the same parameters and
// results file root names.  If temporary file names are not used,
// then each evalComm processor can define the same names without
// need for interprocessor communication.  However, if temporary
// file names are used, then evalCommRank 0 must define the names
// and broadcast them to other evalComm processors.
int eval_comm_rank   = ColinGlobals::processor_id();
bool bcast_flag = ( analysis_servers > 1 &&
    ( (parametersFileName.size() == 0) || (resultsFileName.size() == 0) ) );

if (eval_comm_rank == 0 || !bcast_flag) {
   char ctr_tag[16];
   sprintf(ctr_tag,".%d",id);
   if (parametersFileName.size() == 0) {
      // no user spec ->Unix tmp files as default
      char tmp[256];
      sprintf(tmp,"/tmp/colin_paramFName_%d_XXXXXX",getpid());
      #if defined(__MINGW32__)
      abort();
      #else
      modifiedParamsFileName = mkstemp(tmp);
      #endif
      }
   else {
      modifiedParamsFileName = parametersFileName;
      if (fileTagFlag)
	modifiedParamsFileName += ctr_tag;
   }
   if (resultsFileName.size() == 0) {
      // no user spec ->Unix tmp files as default
      char tmp[256];
      sprintf(tmp,"/tmp/colin_resultsFName_%d_XXXXXX",getpid());
      #if defined(__MINGW32__)
      abort();
      #else
      modifiedResultsFileName = mkstemp(tmp);
      #endif
      }
   else {
      modifiedResultsFileName = resultsFileName;
      if (fileTagFlag)
	modifiedResultsFileName += ctr_tag;
      }
  }

#if 0
  ifdef ACRO_HAVE_MPI
  if (bcast_flag) {
    MPI_Comm eval_comm = parallelLib.evaluation_intra_communicator();
    if (eval_comm_rank == 0) {
      // pack the buffer with root file names for the evaluation
      PackBuffer send_buffer;
      send_buffer << modifiedParamsFileName << modifiedResultsFileName;
      // broadcast buffer length so that other procs can allocate UnPackBuffer
      //int buffer_len = send_buffer.len();
      //parallelLib.bcast(buffer_len, eval_comm);
      // broadcast actual buffer
      parallelLib.bcast(send_buffer, eval_comm);
    }
    else
    {
      // receive length of incoming buffer and allocate space for UnPackBuffer
      //int buffer_len;
      //parallelLib.bcast(buffer_len, eval_comm);
      // receive incoming buffer
      //UnPackBuffer recv_buffer(buffer_len);
      UnPackBuffer recv_buffer(256); // 256 should be plenty for 2 filenames
      parallelLib.bcast(recv_buffer, eval_comm);
      recv_buffer >> modifiedParamsFileName >> modifiedResultsFileName;
    }
  }
#endif
}


//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void AnalysisCode<DomainT,ResponseT,StringT>
	::write_parameters_file(const DomainT& vars, 
                                         const std::vector<int>& asv, 
                                         const int id)
{
//
// If a new evaluation, insert the modified file names into list for use in 
// identifying the proper file names within read_results_file for the case
// of asynch. fn. evals.  If a replacement eval. (e.g., failure capturing 
// with retry or continuation), then overwrite old file names with new ones.
//
std::map<int,std::string>::iterator curr = parametersFName.find(id);
if (curr != parametersFName.end()) {
   //
   // remove old files.  This is only necessary for the tmp file case, since 
   // all other cases (including tagged) will overwrite the old files.  Since
   // we don't want to save tmp files, don't bother to check fileSaveFlag.
   //
   remove(curr->second.data());
   curr->second= modifiedParamsFileName;
   //
   // Do the same for the resultsFName map
   //
   std::map<int,std::string>::iterator curr = resultsFName.find(id);
   remove(curr->second.data());
   curr->second= modifiedResultsFileName;
   }
//
// id does not yet exist, insert new filenames
//
else {
   parametersFName[id] = modifiedParamsFileName;
   resultsFName[id] = modifiedResultsFileName;
   }
//
// Some debuging I/O
//
if (!quietFlag) {
   ucout << "\nFile name list entries at fn. eval. " << id << '\n';
   std::map<int,std::string>::const_iterator currParams = parametersFName.begin();
   std::map<int,std::string>::const_iterator lastParams = parametersFName.begin();
   std::map<int,std::string>::const_iterator currResults = resultsFName.begin();
   while (currParams != lastParams) {
     ucout << currParams->second << " " << currResults->second << " " 
            << currParams->first << '\n';
     currParams++;
     currResults++;
     }
   ucout << std::endl;
   }
//
// Write the parameters file
//
std::ofstream parameter_stream(modifiedParamsFileName.data());
if (!parameter_stream)
   EXCEPTION_MNGR(std::runtime_error,
	"AnalysisCode::write_parameters_file -  cannot create parameters file \"" << modifiedParamsFileName.data() << "\".");
//
// Write out in aprepro format if requested
//
size_t i, asv_len = asv.size();
if (apreproFlag) {
   //
   // TODO - aprerpo IO
   // It isn't clear how to write out generic aprepro formated variable info.
   // The DAKOTA_VARS output really doesn't make any sense for generic
   // DomainT types.
   //
   parameter_stream << "                    { DAKOTA_VARS     = 0 "
                    << " }\n                    { DAKOTA_FNS      = " 
#if defined(CYGWIN)
                    << asv_len << " }\n"; operator << (parameter_stream,vars) << std::endl;
#else
                    << asv_len << " }\n" << vars << std::endl;
#endif
   for (i=0; i<asv_len; i++)
     parameter_stream << "                    { ASV_" << i+1<< "           = "
                      << std::setw(ColinGlobals::write_precision+7) << asv[i] << " }\n";
   }
//
// Write out in standard format
//
else {
   parameter_stream << "                      " 
#if defined(CYGWIN)
                    << asv_len << " functions\n"; operator <<(parameter_stream, vars) << std::endl;
#else
                    << asv_len << " functions\n" << vars << std::endl;
#endif
   for (i=0; i<asv_len; i++)
     parameter_stream << "                     "  << std::setw(ColinGlobals::write_precision+7) 
                      << asv[i] << " ASV_" << i+1 << '\n';

  }
//
// Explicit flush and close added 3/96 to prevent Solaris problem of input 
// filter reading file before the write was completed.
//
parameter_stream.flush();
parameter_stream.close();
}


//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void AnalysisCode<DomainT,ResponseT,StringT>
	::read_results_file(ResponseT& response, const int id)
{
// Retrieve parameters & results file names using fn. eval. id.  A list of
// filenames is used because the names of tmp files must be available here 
// and asynch_recv operations can perform output filtering out of order 
// (which rules out making the file names into attributes of AnalysisCode or
// rebuilding the file name from the root name plus the counter).
std::string& parameters_filename = parametersFName[id];
std::string& results_filename = resultsFName[id];

// If multiple analysis programs are used, then results_filename is tagged
// with program number (to match ForkAnalysisCode::fork_application), e.g.
// results.out.20.2 is the 2nd analysis results from the 20th fn eval.
size_t i;
char prog_num[16];
std::vector<std::string> results_filenames;
if (numPrograms > 1) {
   results_filenames.resize(numPrograms);
   for (i=0; i<numPrograms; i++) {
     int tmp = i+1;
     sprintf(prog_num,".%d",tmp);
     results_filenames[i] = results_filename + prog_num;
     }
   }

// Read the results file(s).  If there's more than one program, then partial
// responses must be overlaid to create the total response.  If an output
// filter is used, then it has the responsibility to perform the overlay and
// map results.out.[eval#].[1->numPrograms] to results.out.[eval#].  If no
// output filter is used, then perform the overlay here.
if ((numPrograms > 1) && (oFilterName == "")) {
#if 0		/// TODO - read results file
   response.reset();
   ResponseT partial_response = response.copy();
   for (i=0; i<numPrograms; i++) {
     std::ifstream recovery_stream(results_filenames[i]);
     if (!recovery_stream)
        EXCEPTION_MNGR(std::runtime_error,
		"AnalysisCode::read_parameters_file -  cannot open results file \"" << results_filenames[i].data() << "\".");
     recovery_stream >> partial_response;
     response.overlay(partial_response);
     }
#endif
   }
else {
   std::ifstream recovery_stream(results_filename.data());
   if (!recovery_stream)
        EXCEPTION_MNGR(std::runtime_error,
		"AnalysisCode::read_parameters_file -  cannot open results file \"" << results_filename.data() << "\".");
   recovery_stream >> response;
   }

// Clean up files based on fileSaveFlag.
if (!fileSaveFlag) {
   if (!quietFlag && verboseFlag) {
      ucout << "Removing " << parameters_filename << " and "
           << results_filename;
      if (numPrograms > 1) {
        if (!oFilterName.size())
           ucout << " and " << results_filename;
        ucout << ".[1-" << numPrograms << ']';
      }
      ucout << '\n';
    }
    remove(parameters_filename.data());
    if (numPrograms == 1 || !oFilterName.size())
      remove(results_filename.data());
    if (numPrograms > 1) {
      for (i=0; i<numPrograms; i++)
        remove(results_filenames[i].data());
    }
  }

// Prevent overwriting of files with reused names for which a file_save
// request has been given.
if (fileSaveFlag && !fileTagFlag && 
    (!parametersFileName.size() || !resultsFileName.size()) ) {
    //
    if (!quietFlag && verboseFlag)
       ucout << "Files with nonunique names will be tagged for file_save:\n";
    char ctr_tag[16];
    sprintf(ctr_tag,".%d",id);
    if (!parametersFileName.size()) {
       std::string new_str = parametersFileName + ctr_tag;
       if (!quietFlag && verboseFlag)
          ucout << "Moving " << parametersFileName << " to " << new_str << '\n';
       rename(parametersFileName.data(), new_str.data());
       }
    if (!resultsFileName.size()) {
       std::string old_str = resultsFileName;
       StringT new_str = resultsFileName + ctr_tag;
       if (numPrograms == 1 || (oFilterName.size() > 0)) {
          if (!quietFlag && verboseFlag)
             ucout << "Moving " << old_str << " to " << new_str << '\n';
          rename(old_str.data(), new_str.data());
          }
       if (numPrograms > 1) { // append program counters to old_str/new_str
          for (i=0; i<numPrograms; i++) {
            int tmp = i+1;
            sprintf(prog_num,".%d",tmp);
            std::string old_tagged_str = old_str + prog_num;
            std::string new_tagged_str = new_str + prog_num;
            if (!quietFlag && verboseFlag)
               ucout << "Moving " << old_tagged_str << " to " << new_tagged_str
                 	<< '\n';
            rename(old_tagged_str.data(), new_tagged_str.data());
	    }
         }
       }
    }
//
// Remove the evaluation which has been processed from the filename lists
//
parametersFName.erase(id);
resultsFName.erase(id);
}

}

#endif
