/*! \file
 
 \brief The AnnotationManager that filters UDDUChains for XAIF.
 
 \authors Michelle Strout
 \version $Id: ManagerAliasMapXAIF.cpp,v 1.18 2005/03/17 21:47:47 mstrout Exp $
 
 Copyright (c) 2002-2005, Rice University <br>
 Copyright (c) 2004-2005, University of Chicago <br>
 Copyright (c) 2006, Contributors <br>
 All rights reserved. <br>
 See ../../../Copyright.txt for details. <br>
 */


/* ==============================================================
 *
 * AliasMapXAIF converts Alias Analysis results into the sets of
 * virtual addreeses needed by OpenADFortTk.
 *
 * OpenADFortTk then finds overlapping or intersecting virtual 
 * locations.
 * =============================================================
 */


//! ==========================================
//! Header files
//! =========================================
#include <OpenAnalysis/XAIF/ManagerAliasMapXAIF.hpp>
#include <Utils/Util.hpp>

#include<stdio.h>
namespace OA {
  namespace XAIF {
    
    
    //! ===========================================
    //! Turn on debug=true, during debugging
    //! ===========================================
    static bool oadebug = false;
    
    
    //! =================================================================
    //! Constructor sets up XAIFIRInterface
    //! Input: XAIFIRInterface
    //! =================================================================
    ManagerAliasMapXAIF::ManagerAliasMapXAIF(OA_ptr<XAIFIRInterface> _ir) 
    : mIR(_ir)
    {
      OA_DEBUG_CTRL_MACRO("DEBUG_ManagerAliasMapXAIF:ALL", oadebug);
    }
    
    
    
    
    //! =================================================================
    //! PerformAnalysis creates setes of virtual addresses.
    //! Input:   Procedure, Alias Analysis Results
    //! Output:  AliasMapXAIF for the procedure
    //! =================================================================
    
    
    OA_ptr<XAIF::AliasMapXAIF> 
    ManagerAliasMapXAIF::performAnalysis( ProcHandle proc, 
                                         OA_ptr<Alias::Interface> alias)
    {
      //! ====================================================
      //! Create empty AliasMapXAIF for the procedure
      //! ====================================================
      OA_ptr<AliasMapXAIF> retAliasMap; retAliasMap = new AliasMapXAIF(proc);
      
      oadebug = 1;
      if(oadebug) {
        std::cout << "The Procedure is " 
        << mIR->toString(proc) 
        << std::endl;
      }
      
      //! ====================================================
      //! Get All Statements in the Procedure 
      //! ====================================================
      
      OA_ptr<OA::IRStmtIterator> sItPtr = mIR->getStmtIterator(proc);
      for ( ; sItPtr->isValid(); (*sItPtr)++) {
        OA::StmtHandle stmt = sItPtr->current();
        
        
        if (oadebug) {
          std::cout << "===========================" 
          << std::endl 
          << "stmt = ";
          mIR->dump(stmt,std::cout);
          std::cout << std::endl;
        }
        
        
        //! =========================================== 
        //! Get All Memory References in the statement 
        //! ===========================================
        
        OA_ptr<MemRefHandleIterator> mrItPtr = mIR->getAllMemRefs(stmt);
        for ( ; mrItPtr->isValid(); (*mrItPtr)++) {
          MemRefHandle memref = mrItPtr->current();
          
          
          if (oadebug) {
            std::cout << "memref = " 
            << mIR->toString(memref) 
            << std::endl;
          }
          
          
          //! ==================================================
          //! Get the AliasTagSet for each memory reference 
          //! ===============================================
          OA_ptr<Alias::AliasTagSet> aTagSet 
          = alias->getAliasTags(memref);
          
          
          OA_ptr<Alias::AliasTagIterator> aTagIter;
          aTagIter = aTagSet->getIterator();
          
          std::set<int> aSet;
          
          for(; aTagIter->isValid(); ++(*aTagIter)) {
            aSet.insert(aTagIter->current().val());
          }
          
          int setId=-1;
          
          //! ==========================================
          //! AddressOf Memory Reference 
          //! ==========================================
          if(aTagSet->size() == 0) { setId=0; }
          else {
            
            //! =========================================
            //! Get Valid SetId for the Memory Reference 
            //! =========================================
            setId = retAliasMap->getMapSetId(aSet);
          }
          if (oadebug) {
            std::cout << "setId = " 
            << setId 
            << std::endl;
          }
          //! =============================================
          //! if can't find valid setid then create a new 
          //! one and map it to a new one 
          //! =============================================
          if (setId==AliasMapXAIF::SET_ID_NONE/*||setId==0*/) { 
            setId = retAliasMap->makeEmptySet();

            
            //! ============================================
            //! Create set of Valid Virtual Addresses 
            //! using AliasTags 
            //! ============================================
            
            OA_ptr<Alias::AliasTagIterator> aIter = aTagSet->getIterator()  ;
            OA_ptr<std::set<LocTuple> > lSet;
            lSet = new std::set<LocTuple>;
            
            for(; aIter->isValid(); ++(*aIter)) {
              Alias::AliasTag aTag = aIter->current();
              LocTuple ltuple 
              = LocTuple(aTag.val(), aTag.val(), aTagSet->isMust());
              lSet->insert(ltuple);
            }
            
            
            //! ============================================
            //! insert all locations into this new set 
            //! ===========================================
            
            retAliasMap->mapLocTupleSet(lSet, setId);
            
            
            //! ==================================================

            
            retAliasMap->mapSetToSetId(aSet, setId);
  
          }   
          //! ==============================================
          //! map top memrefhandle to determined alias map set 
          //! ==============================================
          retAliasMap->mapMemRefToMapSet(memref,setId);

        }
      }
      
      //! ==========================================
      //! Return AliasMapXAIF for the procedure
      //! ========================================== 
      
      return retAliasMap;
    }
    
  } // end of namespace XAIF
} // end of namespace OA

