/*! \file
  
  \brief Declarations of an AnnotationManager that generates an 
         InterReachConsts using the ICFGCSDFSolver.

  \authors Michelle Strout, Barbara Kreaseck
  \version $Id: ManagerICFGCSReachConsts.hpp,v 1.1.2.1 2006/02/10 07:57:31 mstrout Exp $

  Copyright (c) 2002-2004, Rice University <br>
  Copyright (c) 2004, University of Chicago <br>  
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>

*/

#ifndef ManagerICFGCSReachConsts_H
#define ManagerICFGCSReachConsts_H

//--------------------------------------------------------------------
#include <cassert>
#include <vector>

// OpenAnalysis headers
#include <OpenAnalysis/Alias/Interface.hpp>
#include <OpenAnalysis/SideEffect/InterSideEffectInterface.hpp>
#include <OpenAnalysis/IRInterface/ReachConstsIRInterface.hpp>
#include <OpenAnalysis/DataFlow/ICFGCSDFProblem.hpp>
#include <OpenAnalysis/DataFlow/ICFGCSDFSolver.hpp>
#include <OpenAnalysis/DataFlow/IRHandleDataFlowSet.hpp>
#include <OpenAnalysis/DataFlow/ParamBindings.hpp>
#include <OpenAnalysis/IRInterface/ConstValBasicInterface.hpp>
#include <OpenAnalysis/ExprTree/EvalToConstVisitor.hpp>
#include <OpenAnalysis/ReachConsts/InterReachConsts.hpp>
#include <OpenAnalysis/Alias/CCSetPerProc.hpp>

namespace OA {
  namespace ReachConsts {

/*! 
    Creates InterReachConsts, which can be queried for a CSReachConsts for
    each procedure.  
*/
class ManagerICFGCSReachConsts : public virtual DataFlow::ICFGCSDFProblem
{ 
public:
  ManagerICFGCSReachConsts(OA_ptr<ReachConstsIRInterface> _ir);
  ~ManagerICFGCSReachConsts () {}

  OA_ptr<InterReachConsts> 
  performAnalysis(OA_ptr<ICFG::ICFGInterface> icfg,
                  OA_ptr<Alias::Interface> Alias,
                  OA_ptr<Alias::CCSetPerProc> ccResults,
                  DataFlow::DFPImplement algorithm);

private:
  
  //! ========================================================
  //! implementation of ICFGDFProblem interface
  //! ========================================================


  //! --------------------------------------------------------
  //! initialization callbacks
  //! --------------------------------------------------------


  //! ===== Return an initialized top set =====
  OA_ptr<DataFlow::DataFlowSet>  initializeTop(OA_ptr<Alias::CallContext> cc);
  

  //! ===== Should generate an in and out DataFlowSet for node =====
  OA_ptr<DataFlow::DataFlowSet> 
  initializeNodeIN(OA_ptr<ICFG::NodeInterface> n, 
                   OA_ptr<Alias::CallContext> cc);

  OA_ptr<DataFlow::DataFlowSet> 
  initializeNodeOUT(OA_ptr<ICFG::NodeInterface> n, 
                    OA_ptr<Alias::CallContext> cc);
 

  //! --------------------------------------------------------
  //! solver callbacks 
  //! --------------------------------------------------------
 
  //! ===== Ok to modify set1 and set2 ===== 
  OA_ptr<DataFlow::DataFlowSet> 
  meet(OA_ptr<DataFlow::DataFlowSet> set1, 
       OA_ptr<DataFlow::DataFlowSet> set2,
       OA_ptr<Alias::CallContext> cc); 


  OA_ptr<DataFlow::DataFlowSet> 
  transfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in, 
           OA_ptr<Alias::CallContext> cc, OA::StmtHandle stmt);
 



  OA_ptr<DataFlow::DataFlowSet>
  transfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in,
           OA_ptr<Alias::CallContext> cc, OA::CallHandle call);


 
  //! transfer function for the entry node of the given procedure
  //! should manipulate incoming data-flow set in any special ways
  //! for procedure and return outgoing data-flow set for node
  OA_ptr<DataFlow::DataFlowSet> 
  entryTransfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in,
                OA_ptr<Alias::CallContext> cc);



  //! No needed
  OA_ptr<DataFlow::DataFlowSet> 
  exitTransfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> out,
               OA_ptr<Alias::CallContext> cc);



  //! ===== Propagate a data-flow set from caller to callee =====
  OA_ptr<DataFlow::DataFlowSet> 
  callerToCallee(ProcHandle caller, OA_ptr<DataFlow::DataFlowSet> dfset,
                 OA_ptr<Alias::CallContext> cc,
                 CallHandle call, ProcHandle callee);
  

  //! ===== Propagate a data-flow set from callee to caller =====
  OA_ptr<DataFlow::DataFlowSet> 
  calleeToCaller(ProcHandle callee, OA_ptr<DataFlow::DataFlowSet> dfset,
                 OA_ptr<Alias::CallContext> cc,
                 CallHandle call, ProcHandle caller);


  //! ===== Propagate a data-flow set from call node to return node =====
  OA_ptr<DataFlow::DataFlowSet> 
  callToReturn(ProcHandle caller, OA_ptr<DataFlow::DataFlowSet> dfset,
               OA_ptr<Alias::CallContext> cc,
               CallHandle call, ProcHandle callee);

  //! --------------------------------------------------------
  //! solver callbacks for DEBUGGING
  //! --------------------------------------------------------

  //! ===== Only the problem itself knows what type of DataFlowSet is being
  //! used, and how to print it ... ICFGCSSolver does not know that info
  void printDFSet(std::ostream& os, OA_ptr<DataFlow::DataFlowSet>);

  //! ===== ICFGCSSolver cannot print CallContexts ... (no IRInterface)
  //! so, ICFGCSDFProblem needs to do it.
  void printCallContext(std::ostream& os, OA_ptr<Alias::CallContext> cc);

  //! --------------------------------------------------------
  //! helper routines 
  //! --------------------------------------------------------
 
  //! ===== Needed by transfer =====
  OA_ptr<RCPairSet> transferAssignPair(StmtHandle s, CallHandle c,
                                       bool haveStmt,
                                       MemRefHandle mref, ExprHandle expr,
                                       ProcHandle proc,
                                       OA_ptr<RCPairSet> inSet,
                                       OA_ptr<Alias::CallContext> cc);



private:
  OA_ptr<InterReachConsts> mInterReachConsts;
  OA_ptr<ReachConstsIRInterface> mIR;
  OA_ptr<Alias::Interface> mCSAlias;
  OA_ptr<DataFlow::ICFGCSDFSolver> mSolver;
  OA_ptr<ICFG::ICFGInterface> mICFG;
  std::map<ProcHandle,OA_ptr<CSReachConsts> > mCSReachConstsMap;

}; // end of ManagerICFGCSReachConsts.hpp

  } // end of ReachConsts namespace
} // end of OA namespace

#endif
