/*! \file
  
  \brief Implementation of DataFlow::CC2DFSMap

  \author Michelle Strout, Barbara Kreaseck
  \version $Id: CCDFSPairSet.cpp,  Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/


#include "CC2DFSMap.hpp"
#include <OpenAnalysis/Utils/Util.hpp>
#include <OpenAnalysis/Utils/OutputBuilder.hpp>

namespace OA {
  namespace DataFlow {

static bool debug = false;


//! ===============================================================
//! CC2DFSMap methods
//! ===============================================================

CC2DFSMap::CC2DFSMap()
{
  OA_DEBUG_CTRL_MACRO("DEBUG_CC2DFSMap:ALL", debug);
    mMap = new std::map<OA_ptr<Alias::CallContext>, OA_ptr<DataFlowSet> >;
}


//! does not clone, makes a copy
CC2DFSMap::CC2DFSMap(const CC2DFSMap& other)
  :  mMap(other.mMap)
{
}



CC2DFSMap::~CC2DFSMap() { }



OA_ptr<CC2DFSMap> CC2DFSMap::clone() const
{
   OA_ptr<CC2DFSMap> retval;
   retval = new CC2DFSMap();

   std::map<OA_ptr<Alias::CallContext>, OA_ptr<DataFlowSet> >::iterator mapIter;
   mapIter = mMap->begin();
   for ( ; mapIter!=(mMap->end()); ++mapIter) {
     OA_ptr<Alias::CallContext> cc = (mapIter->first)->clone();
     OA_ptr<DataFlowSet> dfs = (mapIter->second)->clone();
     retval->assign(cc,dfs);
   }
   return retval;
}

    // copies, does not clone
CC2DFSMap& CC2DFSMap::operator= (const CC2DFSMap& other)
{
  mMap = other.mMap;
  return *this;
}


void CC2DFSMap::clear() { 
  mMap->clear();
}


int  CC2DFSMap::size() const { 
  return mMap->size();
}



bool CC2DFSMap::isEmpty() const { 
  return (mMap->empty()); 
}


OA_ptr<DataFlowSet> CC2DFSMap::getDFS(OA_ptr<Alias::CallContext> cc) const
{
  OA_ptr<DataFlowSet> retval;
  std::map<OA_ptr<Alias::CallContext>, OA_ptr<DataFlowSet> >::iterator mIter;

  mIter = mMap->find(cc); // assuming cc is not NULL

  if (mIter != mMap->end()) { // then cc is in the map
    retval = mIter->second;
  }
  return retval;
}




void CC2DFSMap::assign(OA_ptr<Alias::CallContext> cc, OA_ptr<DataFlowSet> dfs) { 
  // just copies, does not clone
  // also: replaces any previous association :  dfs will replace current (*mMap)[cc]

  // this is not a UNION, it is an insert.  We assume that the caller
  // knows what they are doing.
  
  /*
  if ( !(getDFS(cc).ptrEqual(0)) ) {
    assert(false && "ERROR?? CC2DFSMap::insert(): CC already in map");
  }
  */
  
  // insert mapping cc -> dfs
  (*mMap)[cc] = dfs;
  
}


void CC2DFSMap::remove(OA_ptr<Alias::CallContext>cc) { 
  
   mMap->erase(cc);

}


//! operator == for a CC2DFSMap CAN NOW rely upon the == operator for the
// underlying set, because the == operator of an element of a CC2DFSMap, 
// namely a CCDFSPair, considers both the CallContext and DataFlowSet
bool CC2DFSMap::operator==(const CC2DFSMap &other) const
{
  // step through other, check for same dfs for cc in *this

  /*  I really can't do the following, right??
    return (mMap == other.mMap);
  */

  if (mMap->size() != other.mMap->size()) {
    return false;
  }

  bool retval = true;

  // if the sizes are the same, shouldn't I only need to step through in one direction??
  std::map<OA_ptr<Alias::CallContext>, OA_ptr<DataFlowSet> >::iterator mIter;
  mIter = other.mMap->begin();
  while (mIter != other.mMap->end()) {
    OA_ptr<DataFlowSet> dfs;
    dfs = getDFS(mIter->first);
    if (dfs.ptrEqual(0)) {
      return false; // cc in other but not in *this
    }
    if (dfs != (mIter->second)) {
      return false; // found non-equal dfs for same cc
    }
    mIter++;
  }
  
  return retval;

}



bool CC2DFSMap::operator !=(const CC2DFSMap &other) const
{
  return (!(*this==other)); 
}

//! operator < for a CC2DFSMap relies upon the < operator for the
// element of a CC2DFSMap, namely a CCDFSPair, which considers only the CallContext
// (not the best < implementation, but may be sufficient for use with sets)
bool CC2DFSMap::operator<(const CC2DFSMap &other) const
{
  if (mMap->size() < other.mMap->size()) {
    return true;
  } else if (mMap->size() > other.mMap->size()) {
    return false;
  }


  // if the sizes are the same, shouldn't I only need to step through in one direction??
  std::map<OA_ptr<Alias::CallContext>, OA_ptr<DataFlowSet> >::iterator mIter1;
  std::map<OA_ptr<Alias::CallContext>, OA_ptr<DataFlowSet> >::iterator mIter2;
  mIter1 = mMap->begin();
  mIter2 = other.mMap->begin();
  while (mIter1 != mMap->end() && mIter2 != other.mMap->end()) {
    CCDFSPair cdPair1(mIter1->first, mIter1->second);
    CCDFSPair cdPair2(mIter2->first, mIter2->second);
    if (cdPair1 < cdPair2) {
      return true;
    } else if (cdPair2 < cdPair1) {
      return false;
    } // else // equal CallContexts, keep looking
    mIter1++;
    mIter2++;
  }
  
  return false;

}



//! ================ Output =========================
void CC2DFSMap::output(IRHandlesIRInterface& ir) const {
     std::cout << "Please call output with alias::Interface"
               << std::endl;
     assert(0);
}


void CC2DFSMap::output(OA::IRHandlesIRInterface &ir,
                       Alias::Interface& alias) const
{
  sOutBuild->objStart("CC2DFSMap");
  
  sOutBuild->listStart();
  OA_ptr<CC2DFSMapIterator> defIter;
  defIter = new CC2DFSMapIterator(*this);
  for (; defIter->isValid(); ++(*defIter)) {
    sOutBuild->listItemStart();
    sOutBuild->fieldStart("CCDFSPair");
    (defIter->current()).output(ir,alias);
    sOutBuild->fieldEnd("CCDFSPair");
    sOutBuild->listItemEnd();
  }
  sOutBuild->listEnd();

  sOutBuild->objEnd("CC2DFSMap");
}



void CC2DFSMap::dump(std::ostream &os, OA_ptr<IRHandlesIRInterface> pIR)
{
  if (mMap->size() > 0) {
    os << "CC2DFSMap Pairs: ";
    OA_ptr<CC2DFSMapIterator> defIter;
    defIter = new CC2DFSMapIterator(*this);
    if (defIter->isValid()) { // handle first CCDFSPair
      (defIter->current()).dump(os, pIR);
      ++(*defIter);
    }
    for (; defIter->isValid(); ++(*defIter)) { // handle rest
      os << ", "; 
      (defIter->current()).dump(os,pIR);
    }
  }
  
  os << std::endl; 
}




//! ===============================
//! CC2DFSMap Iterator 
//! ===============================

CC2DFSMapIterator::CC2DFSMapIterator (const CC2DFSMap& cc2dfsMap) 
{
  mcc2dfsMap = cc2dfsMap.clone();
  assert(!mcc2dfsMap.ptrEqual(0));
  mIter = mcc2dfsMap->mMap->begin();;

}




CC2DFSMapIterator::~CC2DFSMapIterator () {}




void CC2DFSMapIterator::operator ++ () { 
  if (isValid()) mIter++; 
}




//! is the iterator at the end
bool CC2DFSMapIterator::isValid() 
{ 
  return (mIter!=mcc2dfsMap->mMap->end()); 
}




//! return copy of current node in iterator
CCDFSPair CC2DFSMapIterator::current() 
{ 
  assert(isValid());
  return CCDFSPair(mIter->first, mIter->second); 
}




//! reset iterator to beginning of set
void CC2DFSMapIterator::reset() 
{
  mIter = mcc2dfsMap->mMap->begin();
}





  } // end of DataFLow namespace
} // end of OA namespace

