/*! \file
  
  \brief Implementation of MREwContext

  \authors Barbara Kreaseck
  \version $Id$

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include "MREwContext.hpp"

namespace OA {
  namespace Alias {


// *******************************************************
// Beginning of MREwContext code
// *******************************************************


MREwContext::MREwContext(OA_ptr<MemRefExpr> mre, OA_ptr<CallContext> cc)
{
  mMRE = mre;
  mCallContext = cc;
}

MREwContext::MREwContext(OA_ptr<MREwContext> other)
{
  mMRE = other->mMRE;
  mCallContext = other->mCallContext;

  // to clone or not to clone ... =)
  /*
  mMRE = (other->mMRE)->clone();
  mCallContext = (other->mCallContext)->clone();
  */
}

MREwContext::~MREwContext()
{

}

OA_ptr<MemRefExpr> MREwContext::getMRE()
{
  return mMRE;
}

OA_ptr<CallContext> MREwContext::getCallContext()
{
  return mCallContext;
}

bool MREwContext::operator ==(const MREwContext& other) const
{
  if ((*mMRE)==(*(other.mMRE))) { // do I need to deref these?
    if (mCallContext.ptrEqual(0)) {
      if ((other.mCallContext).ptrEqual(0)) {
        return true;
      } else {
        // BK: if one's CallContext is NULL and the other is not NULL for same MRE
        // then shouldn't this assert?  Seems like if we are using NULL for 
        // NamedRefs (local vars), then we shouldn't see a NULL and a nonNULL 
        // for the same MRE
        return false;
      }
    } else {
      if ((other.mCallContext).ptrEqual(0)) {
        // BK: see comment above
        return false;
      } else {
        // both are non-NULL
        if ((*mCallContext)==(*(other.mCallContext))) {
          return true;
        } else {
          return false;
        }
      }
    }
  }
  return false;
}
  
bool MREwContext::operator <(const MREwContext& other) const
{
  if ((*mMRE) < (*(other.mMRE))) {
    return true;
  } 
  else if ((*mMRE) == (*(other.mMRE))) {
    if (mCallContext.ptrEqual(0) || (other.mCallContext).ptrEqual(0)) {
      // at least one is NULL
      if (mCallContext.ptrEqual(0)) {
        if ((other.mCallContext).ptrEqual(0)) {
          // both NULL
          return false;
        } else {
          // mCallContext is NULL, and other is not
          return true; // NULL < any other address
        }
      } else {
        // other is NULL, and mCallContext is not
        return false; // any address ( NOT < ) NULL
      }
    } else {
      // both are non-NULL
      return ((*mCallContext) < (*(other.mCallContext)));
    } 
  }
  return false; 
}

OA_ptr<MREwContext> MREwContext::clone()
{
  OA_ptr<MREwContext> result;
  if (mCallContext.ptrEqual(0)) {
    result = new MREwContext(mMRE->clone(),mCallContext);
  } else {
    result = new MREwContext(mMRE->clone(),mCallContext->clone());
  }
  return result;
}

MREwContext& MREwContext::operator=( MREwContext& other)
{
  // This copies, not clones.  
  mMRE = other.mMRE;
  mCallContext = other.mCallContext; // ok to copy NULL here
  return *this;
}

void MREwContext::output(OA::IRHandlesIRInterface& ir) const
{
  sOutBuild->objStart("MREwContext");
  sOutBuild->fieldStart("mMRE");
  if (mMRE.ptrEqual(0)) {
    sOutBuild->field("mMRE","null");
  } else {
    mMRE->output(ir);
  }
  sOutBuild->fieldEnd("mMRE");
  sOutBuild->fieldStart("mCallContext");
  if (mCallContext.ptrEqual(0)) {
    sOutBuild->field("mCallContext","null");
  } else {
    mCallContext->output(ir);
  }
  sOutBuild->fieldEnd("mCallContext");
  sOutBuild->objEnd("MREwContext");
}

void MREwContext::output(OA::IRHandlesIRInterface& ir,
			 const Interface& aliasResults) const
{
  output(ir);
}

void MREwContext::dump(std::ostream& os,
		       IRHandlesIRInterface& ir,
		       Interface& aliasResults)
{
  os << "[ MREwContext: \n";
  if (mMRE.ptrEqual(0)) {
    os << "\tMemRefExpr(null)\n";
  } else {
    mMRE->dump(os,ir);
  }
  if (mCallContext.ptrEqual(0)) {
    os << "\tCallContext(null)\n";
  } else {
    mCallContext->dump(os,ir);
  }
  os << "\n]";
}

  } // end of Alias Namespace
} // end of OA Namespace
