/*! \file
  
  \brief The AnnotationManager that generates a context-sensitive ActivePerStmt.

  \authors Michelle Strout, Barbara Kreaseck, Priya Malusare

  First the Vary in and out sets are computed for each statement
  through use of the ICFGDFSolver.
  Then the in and out active sets are computed as one pass
  in the performAnalysis method.

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include "ManagerICFGCSVaryActive.hpp"
#include <Utils/Util.hpp>
#include <OpenAnalysis/Utils/OutputBuilderDOT.hpp>
#include <OpenAnalysis/ICFG/ICFG.hpp>


namespace OA {
  namespace Activity {

static bool debug = false;

/*!
*/
ManagerICFGCSVaryActive::ManagerICFGCSVaryActive( 
        OA_ptr<ActivityIRInterface> _ir) : mIR(_ir)
{
    OA_DEBUG_CTRL_MACRO("DEBUG_ManagerICFGCSVaryActive:ALL", debug);
    mCSSolver = new DataFlow::ICFGCSDFSolver(DataFlow::ICFGCSDFSolver::Forward,
                                             *this);
}

OA_ptr<Activity::ActivePerStmt> 
ManagerICFGCSVaryActive::performAnalysis(
        OA_ptr<ICFG::ICFGInterface> icfg,
        OA_ptr<Alias::Interface> alias,
        OA_ptr<Alias::CCSetPerProc> ccResults,
        OA_ptr<InterUseful> interUseful,
        DataFlow::DFPImplement algorithm)
{
  // store results that will be needed in callbacks
  mICFG = icfg;
  
  if (debug) {
    OA_ptr<ICFG::ICFG> debugicfg;
    debugicfg = icfg.convert<ICFG::ICFG>();

    std::cout << "\n=-=-=-=--=--= ICFG output -=-=-=--=-=-=\n";
    debugicfg->output(*mIR);
    OA::OA_ptr<OA::OutputBuilder> outBuild;
    outBuild = new OA::OutputBuilderDOT;
    debugicfg->configOutput(outBuild);
    debugicfg->output(*mIR);
    std::cout << "\n=-=-=-=-=-=-= End of ICFG output -=-=-=-=-=-=\n\n";
  }

  mCSAlias = alias;
  mMaxNumTags = mCSAlias->getMaxAliasTag().val()+1;
  mCCSetPerProc = ccResults;
  mInterUseful = interUseful;
  
  // create an empty ActivePerStmt
  mActive = new ActivePerStmt();
  
  // Precalculate the Defs, Uses, and MustDefs set for each statement
  // FIXME: doing the same thing in ManagerICFGUseful, can we refactor
  // this?
  precalcDefsAndUses(icfg);

  // call iterative data-flow solver for ICFG to get Vary results in mActive
  mCSSolver->solve(icfg,mCCSetPerProc,algorithm);

  // debug output for Vary results
  if (debug) {
    std::cout << "*&*&*&*&*&*& --- CS VARY RESULTS (printed as ActivePerStmt)"
              << "\n\n";
    mActive->output(*mIR);
    std::cout << "*&*&*&*&*&*& --- END of CS VARY RESULTS ---\n\n";
  }
  
  // now calculate active results per stmt from Useful and Vary results
  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
  OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> > usefulIter;
  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > varySet;

  // For each ICFG node:
  OA_ptr<ICFG::NodesIteratorInterface> nodeIter =
    icfg->getICFGNodesIterator();
  for ( ; nodeIter->isValid(); (*nodeIter)++) {
    OA_ptr<ICFG::NodeInterface> icfgNode = nodeIter->currentICFGNode();
    // get alias and active results for current procedure
    ProcHandle proc = icfgNode->getProc();
    OA_ptr<Alias::CallContextSetIterator> ccSetIter;
    ccSetIter = mCCSetPerProc->getCallContextSet(proc);
    
    OA_ptr<CSUseful> useful = mInterUseful->getCSUsefulResults(proc);

    if (icfgNode->getCall() != OA::CallHandle(0)) {

       OA::CallHandle call = icfgNode->getCall();

       // for each CC for this node's proc:
       for (ccSetIter->reset() ; ccSetIter->isValid(); ++(*ccSetIter)) {
         OA_ptr<Alias::CallContext> cc = ccSetIter->current();

         // calculate InActive
         retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
         usefulIter = useful->getInUsefulIterator(call,cc);
         varySet = mActive->getInActiveSet(call,cc);

         for ( ; usefulIter->isValid(); (*usefulIter)++ ) {
           Alias::AliasTag usefulTag = usefulIter->current();
           if(varySet->contains(usefulTag)) {
             retval->insert(usefulTag);
           }
         }
         mActive->copyIntoInActive(call, cc, retval);


         // calculate OutActive
         retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
         usefulIter = useful->getOutUsefulIterator(call,cc);
         varySet = mActive->getOutActiveSet(call,cc);

         for ( ; usefulIter->isValid(); (*usefulIter)++ ) {
           Alias::AliasTag usefulTag = usefulIter->current();
           if(varySet->contains(usefulTag)) {
             retval->insert(usefulTag);
           }
         }
         mActive->copyIntoOutActive(call, cc, retval);
       } // end of CC loop

    } else {

       // For each stmt in the ICFG node:
       OA_ptr<CFG::NodeStatementsIteratorInterface> stmtIter =
         icfgNode->getNodeStatementsIterator();

       // for the first statement InActive and OutActive are
       // manual calculations.

       for( ; stmtIter->isValid() ; ++(*stmtIter)) {

          StmtHandle stmt = stmtIter->current();

          // for each CC for this node's proc:
          for (ccSetIter->reset() ; ccSetIter->isValid(); ++(*ccSetIter)) {
            OA_ptr<Alias::CallContext> cc = ccSetIter->current();
            
            // calculate InActive
            retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
            usefulIter = useful->getInUsefulIterator(stmt,cc);
            varySet = mActive->getInActiveSet(stmt,cc);
            
            for ( ; usefulIter->isValid(); (*usefulIter)++ ) {
              Alias::AliasTag usefulTag = usefulIter->current();
              if(varySet->contains(usefulTag)) {
                retval->insert(usefulTag);
              }
            }
            mActive->copyIntoInActive(stmt, cc, retval);

            // calculate InActive
            retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
            usefulIter = useful->getOutUsefulIterator(stmt,cc);
            varySet = mActive->getOutActiveSet(stmt,cc);

            for ( ; usefulIter->isValid(); (*usefulIter)++ ) {
              Alias::AliasTag usefulTag = usefulIter->current();
              if(varySet->contains(usefulTag)) {
                retval->insert(usefulTag);
              }
            }
            mActive->copyIntoOutActive(stmt, cc, retval);
          } // end cc loop
       } // end stmt loop 
    }
  } // end node loop
  
  mActive->setNumIter(mCSSolver->getNumIter());
    
  if (debug) {
      std:: cout << std::endl << "%+%+%+%+ Vary numIter = " 
       << mCSSolver->getNumIter() 
       << " +%+%+%+%" << std::endl << std::endl;
  }
    
  return mActive;
}

//========================================================
// implementation of ICFGDFProblem interface
//========================================================
//--------------------------------------------------------
// initialization callbacks
//--------------------------------------------------------

/*!
   Data-flow set passed around on the call graph is a set of alias tags.
   The top value for this is an empty set.
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSVaryActive::initializeTop(OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    return retval;
}

OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSVaryActive::initializeNodeIN(OA_ptr<ICFG::NodeInterface> n,
                                          OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    
    return retval;
}

OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSVaryActive::initializeNodeOUT(OA_ptr<ICFG::NodeInterface> n,
                                           OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    
    return retval;
 }


//--------------------------------------------------------
// solver callbacks 
//--------------------------------------------------------
  
//! OK to modify set1 and return it as result, because solver
//! only passes a tempSet in as set1
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSVaryActive::meet(OA_ptr<DataFlow::DataFlowSet> set1, 
                              OA_ptr<DataFlow::DataFlowSet> set2,
                              OA_ptr<Alias::CallContext> cc)
{
    if (debug) {
        std::cout << "-------- ManagerICFGCSVaryActive::meet" << std::endl;
    }

    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;

    if(set1->size() < set2->size()) {
     retval
      = set1->clone().convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
     retval->unionEqu(*set2);
    } else {
     retval
      = set2->clone().convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
     retval->unionEqu(*set1);
    }

    return retval;
}


/*! 
    A helper function that determines active locations, whether
    the previous stmt was active, and which memory references in the
    previous and current stmt are active
*/
OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ManagerICFGCSVaryActive::calculateActive(
        OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> > varyIter,
        OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> > usefulIter,
        StmtHandle stmt, OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);

    if (debug) {
        std::cout << "\tcalculateActive ---------------------" << std::endl;
    }

    // get set of active locations
    varyIter->reset();
    for ( ; varyIter->isValid(); (*varyIter)++ ) {
        Alias::AliasTag varyTag = varyIter->current();
        if (debug) { 
            std::cout << "\t\tinVary loc = ";
            std::cout << varyTag << std::endl;
        }
        usefulIter->reset();
        for ( ; usefulIter->isValid(); (*usefulIter)++ ) {
            Alias::AliasTag usefulTag = usefulIter->current();
            if (debug) { 
                std::cout << "\t\tusefulItier loc = ";
                std::cout << usefulTag << std::endl;
            }
            if (varyTag == usefulTag) {
                retval->insert(varyTag);
                retval->insert(usefulTag);
                if (debug) {
                  std::cout << "\t\tinserting active loc = ";
                  std::cout << varyTag << std::endl;
                  std::cout << "\t\tinserting active loc = ";
                  std::cout << usefulTag << std::endl;
                }
            } 
        }
    }

    return retval;
}



/*! 

  transfer function for VaryActive analysis

  // Kill off anything that is being must defined
  outVary = inVary - MustDefs[stmt]

  // Gen everything being defined if any of the uses are in vary in set
  if (inVary intersect Uses[stmt])!=empty
    outVary = outVary union Defs[stmt]
  else
    outVary = outVary

  To handle multiple assignment pairs, have the following definition 
  for MustDefs. FIXME: don't have that yet
    MustDefs = intersection_over_assign_pairs MustDefs[assign pair]

  Can and do precalculate Defs, Uses, and MustDefs for the statement.
 
  Development notes:
  OK to modify in set and return it again as result because
  solver clones the BB in sets
*/

//! OK to modify in set and return it again as result because
//! solver clones the BB in sets
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSVaryActive::transfer(ProcHandle proc,
                                  OA_ptr<DataFlow::DataFlowSet> in,
                                  OA_ptr<Alias::CallContext> cc,
                                  StmtHandle stmt)
{   

    // ===== convert incoming set to more specific subclass =====
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inVary;
    inVary = in.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    // outVary = inVary
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outVary;
    outVary = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*inVary);

    if (debug) {
        std::cout << "In transfer, stmt (" << mIR->toString(stmt) << ")";
        std::cout << "\n\t";
        cc->dump(std::cout, *mIR);
        std::cout << "\n\tinVary = ";
        inVary->output(*mIR);
        std::cout << std::endl;
    }

    // ===== Store In Vary for the Statement =====
    // temporarily using spots that we will eventually use for
    // active alias tags
    mActive->copyIntoInActive(stmt,cc,inVary);
    
    // wrap stmt and cc
    Alias::StmtHwContext shwc(stmt,cc);

    // Kill off anything that is being must defined
    // outVary = inVary - MustDefs[shwc]
    // note that outVary was initialized to inVary
    if (debug) {
      OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > mustDefSet;
      mustDefSet = mStmtwCToMustDefs[shwc];
      if (mustDefSet.ptrEqual(0)) {
        assert("mustDefSet for StmtwC is ptrEqual(0)" && false);
      } else {
        std::cout << "mustDefSet is not ptrEqual(0): \n";
      }
      mustDefSet->output(*mIR);
    }
    outVary->minusEqu(*(mStmtwCToMustDefs[shwc]));

    // Gen everything being defined if any of the uses are in vary in set
    // if (inVary intersect Uses[stmt])!=empty
    // inVary intersect Uses[stmt]
    inVary->intersectEqu(*(mStmtwCToUses[shwc]));
    if (debug) { 
        std::cout << "\n\tmStmtwCToUses[shwc] = ";
        mStmtwCToUses[shwc]->output(*mIR);
    }
    if (! inVary->isEmpty()) {
        if (debug) {
            std::cout << "InVary has non-empty intersect with uses" 
                      << std::endl;
        }
        // outVary = outVary union Defs[stmt]
        outVary->unionEqu(*(mStmtwCToDefs[shwc]));
        if (debug) { 
            std::cout << "\n\tmStmtwCToDefs[shwc] = ";
            mStmtwCToDefs[shwc]->output(*mIR);
        }
    }

    // ===== Store Out Vary for the Statement =====
    // temporarily using spots that we will eventually use for
    // active alias tags
    mActive->copyIntoOutActive(stmt,cc, outVary);


    if (debug) {
        std::cout << "\toutVary = ";
        outVary->output(*mIR);
    }

    return outVary;
}





//! OK to modify in set and return it again as result because
//! solver clones the BB in sets
OA_ptr<DataFlow::DataFlowSet>
ManagerICFGCSVaryActive::transfer(ProcHandle proc,
                                  OA_ptr<DataFlow::DataFlowSet> in,
                                  OA_ptr<Alias::CallContext> cc,
                                  OA::CallHandle call)
{

 
    //! ===== convert incoming set to more specific subclass =====
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inVary;
    inVary = in.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outVary;
    outVary = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*inVary);

    if (debug) {
        std::cout << "In transfer, call (" << mIR->toString(call) << ")";
        std::cout << "\n\t";
        cc->dump(std::cout, *mIR);
        std::cout << "\n\tinVary = ";
        inVary->output(*mIR);
    }

    // ===== Store In Vary for the CallHandle =====
    // temporarily using spots that we will eventually use for
    // active alias tags
    mActive->copyIntoInActive(call,cc,inVary);
    
    // wrap call and cc
    Alias::CallHwContext chwc(call,cc);
    
    // Kill off anything that is being must defined
    // outVary = inVary - MustDefs[call]
    // note that outVary was initialized to inVary
    outVary->minusEqu(*(mCallwCToMustDefs[chwc]));

    // Gen everything being defined if any of the uses are in vary in set
    // if (inVary intersect Uses[call])!=empty
    // inVary intersect Uses[call]
    inVary->intersectEqu(*(mCallwCToUses[chwc]));
    if (debug) { 
        std::cout << "\n\tmCallwCToUses[chwc] = ";
        mCallwCToUses[chwc]->output(*mIR);
    }
    if (! inVary->isEmpty()) {
        if (debug) {
            std::cout << "InVary has non-empty intersect with uses" 
                      << std::endl;
        }
        // outVary = outVary union Defs[call]
        outVary->unionEqu(*(mCallwCToDefs[chwc]));
        if (debug) { 
            std::cout << "\n\tmCallwCToDefs[chwc] = ";
            mCallwCToDefs[chwc]->output(*mIR);
        }
    }
    
    // ===== Store Out Vary for the CallHandle =====
    // temporarily using spots that we will eventually use for
    // active alias tags
    mActive->copyIntoOutActive(call,cc,outVary);


    if (debug) {
        std::cout << "\toutVary = ";
        outVary->output(*mIR);
    }

    return outVary;
}







/*!
   Will get the indep vars if this procedure has any and add them
   to incoming DataFlowSet
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSVaryActive::entryTransfer(ProcHandle proc, 
                                       OA_ptr<DataFlow::DataFlowSet> in,
                                       OA_ptr<Alias::CallContext> cc)
{
  if (debug) {
    std::cout << "-------- entryTransfer\n";
    std::cout << "\tProc: " << mIR->toString(proc) << "\n\t";
    cc->dump(std::cout,*mIR);
    std::cout << "\n";
  }
    // create retval as copy of in
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inRecast;
    inRecast = in.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*inRecast);

    if (debug) {
      std::cout << "ManagerICFGCSVaryActive::entryTransfer\n\tinRecast set:";
      inRecast->output(*mIR);
      std::cout << endl;
    }

    // get iterator over indep MemRefExpr for procedure
    OA_ptr<MemRefExprIterator> indepIter = mIR->getIndepMemRefExprIter(proc);
    // get alias results for this procedure

    for ( indepIter->reset(); indepIter->isValid(); (*indepIter)++ ) {
      OA_ptr<MemRefExpr> memref = indepIter->current();

      if (debug) {
        std::cout << "    indep memref dump = '";
        memref->dump(std::cout, mIR);
        std::cout << "'\n";
        std::cout << "    indep memref output = '";
        memref->output(*mIR);
        std::cout << "'\n";
      }

      // get may locs for memref
      OA_ptr<Alias::AliasTagSet> aTag = mCSAlias->getAliasTags(memref,cc);

      OA_ptr<Alias::AliasTagIterator> tagIter;
      tagIter = aTag->getIterator();
      for ( ; tagIter->isValid(); ++(*tagIter) ) {
        if (debug) {
          std::cout << "\t\t\tinserting AliasTag  : ";
          std::cout << tagIter->current() << std::endl;
          std::cout << "\n";
        }
        retval->insert(tagIter->current());
      }
    }
    if (debug) {
        std::cout << "\tManagerICFGCSVaryActive, Indep locations for proc "
                  << mIR->toString(proc) << ": " << std::endl;

        retval->output(*mIR);
        std::cout << "----" << std::endl;
    }

    return retval;
}

/*!
   Just pass along out because this won't be called since we are a Forward
   analysis
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSVaryActive::exitTransfer(ProcHandle proc, 
                                      OA_ptr<DataFlow::DataFlowSet> out,
                                      OA_ptr<Alias::CallContext> cc)
{
    assert(0);
    return out;
}


//! Propagate a data-flow set from callee to caller
OA_ptr<DataFlow::DataFlowSet>
ManagerICFGCSVaryActive::calleeToCaller(ProcHandle callee,
                                        OA_ptr<DataFlow::DataFlowSet> dfset,
                                        OA_ptr<Alias::CallContext> cc,
                                        CallHandle call, ProcHandle caller)
{
    //! ======== Just propogate entire data flow set ===========

    //! create retval as copy of dfset
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outRecast;
    outRecast = dfset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*outRecast);
    return retval;
}


//! Propagate a data-flow set from caller to callee
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSVaryActive::callerToCallee(ProcHandle caller,
                                        OA_ptr<DataFlow::DataFlowSet> dfset,
                                        OA_ptr<Alias::CallContext> cc,
                                        CallHandle call, ProcHandle callee)
{

    //! ======== Just propogate entire data flow set ============

    //! create retval as copy of dfset
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outRecast;
    outRecast = dfset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*outRecast);
    return retval;
}



//! Propagate a data-flow set from call node to return node
OA_ptr<DataFlow::DataFlowSet>
ManagerICFGCSVaryActive::callToReturn(ProcHandle caller,
                                      OA_ptr<DataFlow::DataFlowSet> dfset,
                                      OA_ptr<Alias::CallContext> cc,
                                      CallHandle call, ProcHandle callee)
{

    //! ====== Dont need to pass anything because callerToCallee will take care =====

    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    return retval;
}


void 
ManagerICFGCSVaryActive::printCallContext(std::ostream& os, 
                                          OA_ptr<Alias::CallContext> cc)
{
  cc->dump(os,*mIR);
}


void 
ManagerICFGCSVaryActive::printDFSet(std::ostream& os, 
                                    OA_ptr<DataFlow::DataFlowSet> dfs)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfsRecast;
    dfsRecast = dfs->clone().convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    dfsRecast->output(*mIR);
}

//-------------------------------------------------------------
// Helper routines
//-------------------------------------------------------------

//! Given an iterator over a set of assignment pairs return all the
//! may def alias tags.
OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ManagerICFGCSVaryActive::calcDefs(OA_ptr<OA::AssignPairIterator> aPairIter,
                                  OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);

    if (debug) { std::cout << "In ManagerICFGCSVaryActive::calcDefs" << std::endl;}
    // ===== Iterate over the AssignPairs ===== 
    for( ; aPairIter->isValid(); ++(*aPairIter)) {
        if (debug) { std::cout << "\tfound assignPair" << std::endl;}

        // ======== Get the LHS (Defs) of Assignment ========
        OA::MemRefHandle memref = aPairIter->currentTarget();

        // ===== Get Defs AliasTagSet ========
        OA_ptr<Alias::AliasTagSet> aSet = mCSAlias->getAliasTags(memref,cc);
        
        // union it in with retval
        OA_ptr<DataFlow::DataFlowSet> temp = aSet->getDataFlowSet();
        OA_ptr<DataFlow::DataFlowSetImpl<Alias::AliasTag> > tagset;
        tagset = temp.convert<DataFlow::DataFlowSetImpl<Alias::AliasTag> >();
        DataFlow::DFSetBitImpl<Alias::AliasTag> bitset(mMaxNumTags,tagset->getSet());
        retval->unionEqu(bitset);
    }
    return retval;
}

//! Given an iterator over a set of assignment pairs return all the
//! must def alias tags.
OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ManagerICFGCSVaryActive::calcMustDefs(OA_ptr<OA::AssignPairIterator> aPairIter,
                                      OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    // start with universal because going to intersect in all must defs
    retval->setUniversal();

    if (debug) { 
      std::cout << "In ManagerICFGCSVaryActive::calcMustDefs" << std::endl;
    }

    // ===== Iterate over the AssignPairs ===== 
    for( ; aPairIter->isValid(); ++(*aPairIter)) {

        // ======== Get the LHS (Defs) of Assignment ========
        OA::MemRefHandle memref = aPairIter->currentTarget();

        // ===== Get Defs AliasTagSet ========
        OA_ptr<Alias::AliasTagSet> aSet = mCSAlias->getAliasTags(memref,cc);
        
        // intersect it in with retval if these are must defs
        if (aSet->isMust()) {
            // union it in with retval
            OA_ptr<DataFlow::DataFlowSet> temp = aSet->getDataFlowSet();
            OA_ptr<DataFlow::DataFlowSetImpl<Alias::AliasTag> > tagset;
            tagset = temp.convert<DataFlow::DataFlowSetImpl<Alias::AliasTag> >();
            DataFlow::DFSetBitImpl<Alias::AliasTag> bitset(mMaxNumTags,tagset->getSet());
            retval->intersectEqu(bitset);
        }
    }
    // if there were no must defs, then we need to create an empty set
    if (retval->isUniversalSet()) {
        retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    }
    return retval;
}

//! Given an iterator over a set of assignment pairs return all the
//! use alias tags in the rhs expression.
//! FIXME: what about the uses in the subexpressions for def?
//! FIXME: Also what happened to the concept of differentiable uses?
OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ManagerICFGCSVaryActive::calcUses(OA::OA_ptr<OA::AssignPairIterator> aPairIter,
                                  OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);

    if (debug) { std::cout << "In ManagerICFGCSVaryActive::calcUses" << std::endl;}
    // ===== Iterate over the AssignPairs ===== 
    for( ; aPairIter->isValid(); ++(*aPairIter)) {
        if (debug) { std::cout << "\tfound assignPair" << std::endl;}

        //! ======== Get the RHS (Uses) of Assignment ========
        OA::ExprHandle expr = aPairIter->currentSource();
        
        //! ========= Evaluate Expression Tree =========
        OA_ptr<ExprTree> eTree = mIR->getExprTree(expr);

        CollectMREVisitor evalVisitor;
        eTree->acceptVisitor(evalVisitor);

        // ======= Collect all use MREs from ExprTree =========
        OA::set<OA_ptr<MemRefExpr> > mSet;
        mSet = evalVisitor.getMemRef();
        // FIXME: ok this is a little weird
        OA::OA_ptr<std::set<OA_ptr<MemRefExpr> > > mSetPtr;
        mSetPtr = new std::set<OA_ptr<MemRefExpr> >(mSet);

        OA_ptr<MemRefExprIterator> mreIter;
        mreIter = new OA::MemRefExprIterator(mSetPtr);

        // ===== Iterate over the Use MRE =====
        for( ; mreIter->isValid(); ++(*mreIter) ) {
            OA::OA_ptr<OA::MemRefExpr> mre;
            mre = mreIter->current();

            // === ignore AddressOf because it is not a Use ===
            if(mre->isaAddressOf()) { continue; }

            //! ===== Get the AliasTagSet for Use MREs =====
            OA::OA_ptr<OA::Alias::AliasTagSet> mretagSet;

            if (debug) {
              OA::OA_ptr<OA::MemRefExpr> mre;
              mre = mreIter->current();
              std::cout << "    use mre dump = '";
              mre->dump(std::cout, mIR);
              std::cout << "'\n";
              std::cout << "    use mre output = '";
              mre->output(*mIR);
              std::cout << "'\n";
            }

            mretagSet = mCSAlias->getAliasTags(mreIter->current(),cc);

            OA_ptr<DataFlow::DataFlowSet> temp = mretagSet->getDataFlowSet();
            OA_ptr<DataFlow::DataFlowSetImpl<Alias::AliasTag> > tagset;
            tagset = temp.convert<DataFlow::DataFlowSetImpl<Alias::AliasTag> >();
            DataFlow::DFSetBitImpl<Alias::AliasTag> bitset(mMaxNumTags,tagset->getSet());
            retval->unionEqu(bitset); 

        } // end mreIter
 
    }
    return retval;
}

void 
ManagerICFGCSVaryActive::precalcDefsAndUses(OA_ptr<ICFG::ICFGInterface> icfg)
{
  // iterate over ICFG nodes
  OA_ptr<ICFG::NodesIteratorInterface> nodeIter;
  nodeIter = icfg->getICFGNodesIterator();
  for (; nodeIter->isValid(); ++(*nodeIter)) {
    OA_ptr<ICFG::NodeInterface> node = nodeIter->currentICFGNode();
    ProcHandle proc = node->getProc();
    OA_ptr<Alias::CallContextSetIterator> ccSetIter;
    ccSetIter = mCCSetPerProc->getCallContextSet(proc);
    if (debug) {
      std::cout << "\nprecalc for Node: " << node->getId() << "\n";
    }
    
    // Do a different initialization based on the ICFG node type
    if(node->getType()==ICFG::CALL_NODE) {
        CallHandle call = node->getCall();

        // ======= Get AssignPairs for the call ========
        OA::OA_ptr<OA::AssignPairIterator> aPairIter;
        aPairIter = mIR->getAssignPairIterator(call);
        
        // for each CC for this node's proc:
        for (ccSetIter->reset() ; ccSetIter->isValid(); ++(*ccSetIter)) {
          OA_ptr<Alias::CallContext> cc = ccSetIter->current();
          
          //wrap call with cc
          Alias::CallHwContext chwc(call,cc);

          mCallwCToDefs[chwc] = calcDefs(aPairIter,cc);
          aPairIter = mIR->getAssignPairIterator(call);
          //aPairIter->reset();
          mCallwCToMustDefs[chwc] = calcMustDefs(aPairIter,cc);
          aPairIter = mIR->getAssignPairIterator(call);
          //aPairIter->reset();
          mCallwCToUses[chwc] = calcUses(aPairIter,cc);
        }

    } else {
        // iterate over the statements in each node
        OA_ptr<CFG::NodeStatementsIteratorInterface> stmtIterPtr 
            = node->getNodeStatementsIterator();
        for (; stmtIterPtr->isValid(); ++(*stmtIterPtr)) {
            OA::StmtHandle stmt = stmtIterPtr->current();

            // for each CC for this node's proc:
            for (ccSetIter->reset() ; ccSetIter->isValid(); ++(*ccSetIter)) {
              OA_ptr<Alias::CallContext> cc = ccSetIter->current();
              
              //wrap stmt with cc
              Alias::StmtHwContext shwc(stmt,cc);

              if (debug) {
                std::cout << "\n";
                shwc.output(*mIR);
              }
            
              // ======= Get AssignPairs for the Statement ========
              OA::OA_ptr<OA::AssignPairIterator> aPairIter;

              aPairIter = mIR->getAssignPairIterator(stmt);
              mStmtwCToDefs[shwc] = calcDefs(aPairIter,cc);
              aPairIter = mIR->getAssignPairIterator(stmt);
              mStmtwCToMustDefs[shwc] = calcMustDefs(aPairIter,cc);
              aPairIter = mIR->getAssignPairIterator(stmt);
              mStmtwCToUses[shwc] = calcUses(aPairIter,cc);

              if (debug) {
                assert (!(mStmtwCToDefs[shwc]).ptrEqual(0));
                assert (!(mStmtwCToMustDefs[shwc]).ptrEqual(0));
                assert (!(mStmtwCToUses[shwc]).ptrEqual(0));
              }
            }
        }
    }
  }

}



  } // end of namespace Activity
} // end of namespace OA
