/*! \file
  
  \brief The AnnotationManager that generates InterUseful using ICFGCSDFSolver.

  \authors Michelle Strout
  \version $Id: ManagerICFGUseful.cpp,v 1.2 2005/06/10 02:32:02 mstrout Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include "ManagerICFGCSUseful.hpp"
#include <Utils/Util.hpp>


namespace OA {
  namespace Activity {

static bool debug = false;

/*!
*/
ManagerICFGCSUseful::ManagerICFGCSUseful( OA_ptr<ActivityIRInterface> _ir) : mIR(_ir)
{
  OA_DEBUG_CTRL_MACRO("DEBUG_ManagerICFGCSUseful:ALL", debug);
  mCSSolver = new DataFlow::ICFGCSDFSolver(DataFlow::ICFGCSDFSolver::Backward,*this);
}


OA_ptr<Activity::InterUseful> 
ManagerICFGCSUseful::performAnalysis(
        OA_ptr<ICFG::ICFGInterface> icfg,
        OA_ptr<Alias::Interface> alias,
        OA_ptr<Alias::CCSetPerProc> ccResults,
        DataFlow::DFPImplement algorithm)
{

  // store results that will be needed in callbacks
  mICFG = icfg;
  mCSAlias = alias;
  mCCSetPerProc = ccResults;
  mMaxNumTags = mCSAlias->getMaxAliasTag().val()+1; 

  // create an empty InterUseful
  mInterUseful = new InterUseful();

  // Precalculate the Defs, Uses, and MustDefs set for each statement
  precalcDefsAndUses(icfg);
  
  // call iterative data-flow solver for ICFG
  mCSSolver->solve(icfg,ccResults,algorithm);

  if (debug) {
      std::cout << std::endl << "%+%+%+%+ CSUseful numIter = " 
                << mCSSolver->getNumIter() 
                << " +%+%+%+%" << std::endl << std::endl;
  }

  mInterUseful->setNumIter(mCSSolver->getNumIter());    

  // assign CSUseful results for each procedure into InterUseful
  std::map<ProcHandle,OA_ptr<CSUseful> >::iterator mapIter;
  for (mapIter=mCSUsefulMap.begin(); mapIter!=mCSUsefulMap.end(); mapIter++) {
    mInterUseful->mapProcToCSUseful(mapIter->first,mapIter->second);
  }

  return mInterUseful;
}

//========================================================
// implementation of ICFGCSDFProblem interface
//========================================================
//--------------------------------------------------------
// initialization callbacks
//--------------------------------------------------------

/*!
   Data-flow set passed around on the call graph is an
   DataFlow::DFSetBitImpl<Alias::AliasTag>.  
   The top value for this is an empty set.
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSUseful::initializeTop(OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    return retval;
}

OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSUseful::initializeNodeIN(OA_ptr<ICFG::NodeInterface> n,
                                      OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    
    return retval;
}

OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSUseful::initializeNodeOUT(OA_ptr<ICFG::NodeInterface> n,
                                       OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    return retval;
 }


//--------------------------------------------------------
// solver callbacks 
//--------------------------------------------------------
  
/*! Useful is a backward MAY analysis, so meet is union.
    Unions set of useful alias tags.
 */
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSUseful::meet(OA_ptr<DataFlow::DataFlowSet> set1, 
                          OA_ptr<DataFlow::DataFlowSet> set2,
                          OA_ptr<Alias::CallContext> cc)
{
    if (debug) {
        std::cout << "-------- ManagerICFGCSUseful::meet" << std::endl;
    }

    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;

    if(set1->size() < set2->size()) {
     retval
      = set1->clone().convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
     retval->unionEqu(*set2);
    } else {
     retval
      = set2->clone().convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
     retval->unionEqu(*set1);
    }

    return retval;
}

/*! 

  transfer function for useful analysis

  // Kill
  inUseful = outUseful - MustDefs[stmt]

  // Gen
  if (outUseful intersect Defs[stmt])!=empty
    inUseful = inUseful union Uses[stmt]
  else
    inUseful = inUseful

  To handle multiple assignment pairs, have the following definition 
  for MustDefs.
    MustDefs = intersection_over_assign_pairs MustDefs[assign pair]

  Can and do precalculate Defs, Uses, and MustDefs for the statement.
 
  Development notes:
  OK to modify in set and return it again as result because
  solver clones the BB in sets
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSUseful::transfer(ProcHandle proc,
                              OA_ptr<DataFlow::DataFlowSet> out,
                              OA_ptr<Alias::CallContext> cc,
                              OA::StmtHandle stmt)
{   

    // ===== convert incoming set to more specific subclass =====
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outUseful;
    outUseful = out.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();

    // inUseful = outUseful
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inUseful;
    inUseful = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*outUseful);

    if (debug) {
        std::cout << "In transfer, stmt (" << mIR->toString(stmt) << ")";
        std::cout << "\n\t";
        cc->dump(std::cout, *mIR);
        std::cout << "\n\toutUseful = ";
        outUseful->output(*mIR);
        std::cout << std::endl;
    }

    // ===============================================================
    // store outUseful set (below stmt) for stmt into procedure's
    // Useful result
    // ===============================================================
    if (mCSUsefulMap[proc].ptrEqual(0)) {
      mCSUsefulMap[proc] = new CSUseful(proc);
    }
    mCSUsefulMap[proc]->copyIntoOutUseful(stmt,cc,outUseful);

    // wrap stmt and cc
    Alias::StmtHwContext shwc(stmt,cc);

    // Kills
    // inUseful = outUseful - MustDefs[stmt]
    // note that inUseful was initialized to outUseful
    inUseful->minusEqu(*(mStmtwCToMustDefs[shwc]));

    // outUseful intersect Defs[stmt]
    outUseful->intersectEqu(*(mStmtwCToDefs[shwc]));
    if (debug) { 
        std::cout << "\n\tmStmtwCToDefs[shwc] = ";
        mStmtwCToDefs[shwc]->output(*mIR);
    }

    if (! outUseful->isEmpty()) {
        if (debug) {
            std::cout << "OutUseful has non-empty intersect with defs" 
                      << std::endl;
        }
        // inUseful = inUseful union Uses[stmt]
        inUseful->unionEqu(*(mStmtwCToUses[shwc]));
        if (debug) { 
            std::cout << "\n\tmStmtwCToUses[stmt] = ";
            mStmtwCToUses[shwc]->output(*mIR);
        }
    }

    // store inUseful (above stmt) for stmt into procedure's Useful results
    mCSUsefulMap[proc]->copyIntoInUseful(stmt,cc,inUseful);

    if (debug) {
        std::cout << "\tinUseful = ";
        inUseful->output(*mIR);
    }

    return inUseful;
}





OA_ptr<DataFlow::DataFlowSet>
ManagerICFGCSUseful::transfer(ProcHandle proc,
                              OA_ptr<DataFlow::DataFlowSet> in,
                              OA_ptr<Alias::CallContext> cc,
                              OA::CallHandle call) 
{

    // ===== convert incoming set to more specific subclass =====
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outUseful;
    outUseful = in.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inUseful;
    inUseful = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*outUseful);

    if (debug) {
        std::cout << "In transfer, call (" << mIR->toString(call) << ")";
        std::cout << "\n\t";
        cc->dump(std::cout, *mIR);
        std::cout << "\n\toutUseful = ";
        outUseful->output(*mIR);
    }

    // ===============================================================
    // store outUseful set (below call) for call into procedure's
    // Useful result
    // ===============================================================
    if (mCSUsefulMap[proc].ptrEqual(0)) {
      mCSUsefulMap[proc] = new CSUseful(proc);
    }
    mCSUsefulMap[proc]->copyIntoCallOutUseful(call,cc,outUseful);

    // wrap call and cc
    Alias::CallHwContext chwc(call,cc);
    
    // Kills
    // inUseful = inUseful - MustDefs[call]
    inUseful->minusEqu(*(mCallwCToMustDefs[chwc]));

    // outUseful intersect Defs[call]
    outUseful->intersectEqu(*(mCallwCToDefs[chwc]));
    if (debug) { 
        std::cout << "\n\tmCallwCToDefs[chwc] = ";
        mCallwCToDefs[chwc]->output(*mIR);
    }

    if (! outUseful->isEmpty()) {
        if (debug) {
            std::cout << "OutUseful has non-empty intersect with defs" 
                      << std::endl;
        }
        // inUseful = outUseful union Uses[call]
        inUseful->unionEqu(*(mCallwCToUses[chwc]));
    }

    if (debug) {
        std::cout << "\tinUseful = ";
        inUseful->output(*mIR);
    }

    // store inUseful (above call) for call into procedure's Useful results
    mCSUsefulMap[proc]->copyIntoCallInUseful(call,cc,inUseful);

    return inUseful;
}


/*!
   Will get the dep vars if this procedure has any and add them
   to incoming DFSet
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSUseful::exitTransfer(ProcHandle proc, 
                                  OA_ptr<DataFlow::DataFlowSet> out,
                                  OA_ptr<Alias::CallContext> cc)
{
  if (debug) {
    std::cout << "-------- exitTransfer\n";
    std::cout << "\tProc: " << mIR->toString(proc) << "\n\t";
    cc->dump(std::cout,*mIR);
    std::cout << "\n";
  }
  // create retval as copy of out
  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outRecast;
  outRecast = out.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
  retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*outRecast);
  
  // get iterator over dep MemRefExpr for procedure
  OA_ptr<MemRefExprIterator> depIter = mIR->getDepMemRefExprIter(proc);
  
  if (mCSUsefulMap[proc].ptrEqual(0)) {
    mCSUsefulMap[proc] = new CSUseful(proc);
  }
  for ( depIter->reset(); depIter->isValid(); (*depIter)++ ) {
    OA_ptr<MemRefExpr> memref = depIter->current();
    
    if (debug) {
      std::cout << "    dep memref dump = '";
      memref->dump(std::cout, mIR);
      std::cout << "'\n";
      std::cout << "    dep memref output = '";
      memref->output(*mIR);
      std::cout << "'\n";
    }
    
    // get may locs for memref
    OA_ptr<Alias::AliasTagSet> aTag = mCSAlias->getAliasTags(memref,cc);
    
    OA_ptr<Alias::AliasTagIterator> tagIter;
    tagIter = aTag->getIterator();
    for ( ; tagIter->isValid(); ++(*tagIter)) {
      if (debug) {
        std::cout << "        found dep AliasTag dump: '";
        std::cout << tagIter->current() ;
        std::cout << "'\n";
      }
      if (debug) {
        std::cout << "BEFORE inserting " << tagIter->current()
                  << " into retval, retval contains:\n";
        retval->output(*mIR);
      }
      
      retval->insert(tagIter->current());
      
      if (debug) {
        std::cout << "AFTER inserting " << tagIter->current()
                  << " into retval, retval contains:\n";
        retval->output(*mIR);
      }
      
      //mCSUsefulMap[proc]->insertDepTag(tagIter->current(),mMaxNumTags);
    }
  } // MemRefExpr Iter end
    
  if (debug) {
    std::cout << "\tManagerICFGCSUseful, Dep locations for proc "
              << mIR->toString(proc) << ": " << std::endl;
    retval->output(*mIR);
    std::cout << "----" << std::endl;
  }
  
  return retval;
}

/*!
   Just pass along out because this won't be called since we are a Backward
   analysis
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSUseful::entryTransfer(ProcHandle proc, 
                                   OA_ptr<DataFlow::DataFlowSet> in,
                                   OA_ptr<Alias::CallContext> cc)
{
    assert(0);
    return in;
}



//! ===================================================================
//! Propagate a data-flow set from caller to callee
//! It works on 3 parts 
//! 1. Assignment Pair evaluation and transfer
//! 2. Nonlocal transfer
//! 3. Local pass by reference transfer
//! ===================================================================


OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSUseful::calleeToCaller(ProcHandle caller,
                                    OA_ptr<DataFlow::DataFlowSet> dfset,
                                    OA_ptr<Alias::CallContext> cc,
                                    CallHandle call, ProcHandle callee)
{
    //! ===== Just pass the entire dataflow set =====

    //! create retval as copy of dfset
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outRecast;
    outRecast = dfset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*outRecast);
    return retval;
}
  



//! Propagate a data-flow set from callee to caller
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSUseful::callerToCallee(ProcHandle callee,
                                    OA_ptr<DataFlow::DataFlowSet> dfset, 
                                    OA_ptr<Alias::CallContext> cc,
                                    CallHandle call, ProcHandle caller)
{
    //! ===== Just pass the entire dataflow set =====

    //! create retval as copy of dfset
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outRecast;
    outRecast = dfset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*outRecast);
    return retval;
}


//! Propagate a data-flow set from call node to return node
OA_ptr<DataFlow::DataFlowSet>
ManagerICFGCSUseful::callToReturn(ProcHandle caller,
                                  OA_ptr<DataFlow::DataFlowSet> dfset, 
                                  OA_ptr<Alias::CallContext> cc,
                                  CallHandle call, ProcHandle callee)
{
    // ===== No need to Pass Anything because calleeToCaller will take care =====

    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    return retval;
}


void 
ManagerICFGCSUseful::printCallContext(std::ostream& os, 
                                      OA_ptr<Alias::CallContext> cc)
{
  cc->dump(os,*mIR);
}


void 
ManagerICFGCSUseful::printDFSet(std::ostream& os, 
                                OA_ptr<DataFlow::DataFlowSet> dfs)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfsRecast;
    dfsRecast = dfs->clone().convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    dfsRecast->output(*mIR);
}

//-------------------------------------------------------------
// Helper routines
//-------------------------------------------------------------

//! Given an iterator over a set of assignment pairs return all the
//! may def alias tags.
OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ManagerICFGCSUseful::calcDefs(OA::OA_ptr<OA::AssignPairIterator> aPairIter,
                              OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);

    if (debug) { std::cout << "In ManagerICFGCSUseful::calcDefs" << std::endl;}
    // ===== Iterate over the AssignPairs ===== 
    for( ; aPairIter->isValid(); ++(*aPairIter)) {
        if (debug) { std::cout << "\tfound assignPair" << std::endl;}

        // ======== Get the LHS (Defs) of Assignment ========
        OA::MemRefHandle memref = aPairIter->currentTarget();

        // ===== Get Defs AliasTagSet ========
        OA_ptr<Alias::AliasTagSet> aSet = mCSAlias->getAliasTags(memref,cc);
        
        // union it in with retval
        OA_ptr<DataFlow::DataFlowSet> temp = aSet->getDataFlowSet();
        OA_ptr<DataFlow::DataFlowSetImpl<Alias::AliasTag> > tagset;
        tagset = temp.convert<DataFlow::DataFlowSetImpl<Alias::AliasTag> >();
        DataFlow::DFSetBitImpl<Alias::AliasTag> bitset(mMaxNumTags,tagset->getSet());
        retval->unionEqu(bitset);
    }
    return retval;
}

//! Given an iterator over a set of assignment pairs return all the
//! must def alias tags.
OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ManagerICFGCSUseful::calcMustDefs(OA::OA_ptr<OA::AssignPairIterator> aPairIter,
                                  OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    // start with universal because going to intersect in all must defs
    retval->setUniversal();

    // ===== Iterate over the AssignPairs ===== 
    for( ; aPairIter->isValid(); ++(*aPairIter)) {

        // ======== Get the LHS (Defs) of Assignment ========
        OA::MemRefHandle memref = aPairIter->currentTarget();

        // ===== Get Defs AliasTagSet ========
        OA_ptr<Alias::AliasTagSet> aSet = mCSAlias->getAliasTags(memref,cc);
        
        // intersect it in with retval if these are must defs
        if (aSet->isMust()) {
            // intersect it in with retval
            OA_ptr<DataFlow::DataFlowSet> temp = aSet->getDataFlowSet();
            OA_ptr<DataFlow::DataFlowSetImpl<Alias::AliasTag> > tagset;
            tagset = temp.convert<DataFlow::DataFlowSetImpl<Alias::AliasTag> >();
            DataFlow::DFSetBitImpl<Alias::AliasTag> bitset(mMaxNumTags,tagset->getSet());
            retval->intersectEqu(bitset);
        }
    }
    // if there were no must defs, then we need to create an empty set
    if (retval->isUniversalSet()) {
        retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);
    }
    return retval;
}

//! Given an iterator over a set of assignment pairs return all the
//! use alias tags in the rhs expression.
//! FIXME: what about the uses in the subexpressions for def?
//! FIXME: Also what happened to the concept of differentiable uses?
OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ManagerICFGCSUseful::calcUses(OA::OA_ptr<OA::AssignPairIterator> aPairIter,
                              OA_ptr<Alias::CallContext> cc)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
    retval = new DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxNumTags);

    if (debug) { std::cout << "In ManagerICFGCSUseful::calcUses" << std::endl;}
    // ===== Iterate over the AssignPairs ===== 
    for( ; aPairIter->isValid(); ++(*aPairIter)) {
        if (debug) { std::cout << "\tfound assignPair" << std::endl;}

        //! ======== Get the RHS (Uses) of Assignment ========
        OA::ExprHandle expr = aPairIter->currentSource();
        
        //! ========= Evaluate Expression Tree =========
        OA_ptr<ExprTree> eTree = mIR->getExprTree(expr);

        CollectMREVisitor evalVisitor;
        eTree->acceptVisitor(evalVisitor);

        // ======= Collect all use MREs from ExprTree =========
        OA::set<OA_ptr<MemRefExpr> > mSet;
        mSet = evalVisitor.getMemRef();
        // FIXME: ok this is a little weird
        OA::OA_ptr<std::set<OA_ptr<MemRefExpr> > > mSetPtr;
        mSetPtr = new std::set<OA_ptr<MemRefExpr> >(mSet);

        OA_ptr<MemRefExprIterator> mreIter;
        mreIter = new OA::MemRefExprIterator(mSetPtr);

        // ===== Iterate over the Use MRE =====
        for( ; mreIter->isValid(); ++(*mreIter) ) {
            OA::OA_ptr<OA::MemRefExpr> mre;
            mre = mreIter->current();

            // === ignore AddressOf because it is not a Use ===
            if(mre->isaAddressOf()) { continue; }

            //! ===== Get the AliasTagSet for Use MREs =====
            OA::OA_ptr<OA::Alias::AliasTagSet> mretagSet;
            mretagSet = mCSAlias->getAliasTags(mreIter->current(),cc);

            OA_ptr<DataFlow::DataFlowSet> temp = mretagSet->getDataFlowSet();
            OA_ptr<DataFlow::DataFlowSetImpl<Alias::AliasTag> > tagset;
            tagset = temp.convert<DataFlow::DataFlowSetImpl<Alias::AliasTag> >();
            DataFlow::DFSetBitImpl<Alias::AliasTag> bitset(mMaxNumTags,tagset->getSet());
            retval->unionEqu(bitset); 

        } // end mreIter
 
    }
    return retval;
}

void ManagerICFGCSUseful::precalcDefsAndUses(OA_ptr<ICFG::ICFGInterface> icfg)
{
  // iterate over ICFG nodes
  OA_ptr<ICFG::NodesIteratorInterface> nodeIter;
  nodeIter = icfg->getICFGNodesIterator();
  for (; nodeIter->isValid(); ++(*nodeIter)) {
    OA_ptr<ICFG::NodeInterface> node = nodeIter->currentICFGNode();
    ProcHandle proc = node->getProc();
    OA_ptr<Alias::CallContextSetIterator> ccSetIter;
    ccSetIter = mCCSetPerProc->getCallContextSet(proc);
    
    // Do a different initialization based on the ICFG node type
    if(node->getType()==ICFG::CALL_NODE) {
        CallHandle call = node->getCall();

        // ======= Get AssignPairs for the call ========
        OA::OA_ptr<OA::AssignPairIterator> aPairIter;
        aPairIter = mIR->getAssignPairIterator(call);
        
        // for each CC for this node's proc:
        for (ccSetIter->reset() ; ccSetIter->isValid(); ++(*ccSetIter)) {
          OA_ptr<Alias::CallContext> cc = ccSetIter->current();
          
          //wrap call with cc
          Alias::CallHwContext chwc(call,cc);

          mCallwCToDefs[chwc] = calcDefs(aPairIter,cc);
          aPairIter = mIR->getAssignPairIterator(call);
          //aPairIter->reset();
          mCallwCToMustDefs[chwc] = calcMustDefs(aPairIter,cc);
          aPairIter = mIR->getAssignPairIterator(call);
          //aPairIter->reset();
          mCallwCToUses[chwc] = calcUses(aPairIter,cc);
        }

    } else {
        // iterate over the statements in each node
        OA_ptr<CFG::NodeStatementsIteratorInterface> stmtIterPtr 
            = node->getNodeStatementsIterator();
        for (; stmtIterPtr->isValid(); ++(*stmtIterPtr)) {
            OA::StmtHandle stmt = stmtIterPtr->current();

            // for each CC for this node's proc:
            for (ccSetIter->reset() ; ccSetIter->isValid(); ++(*ccSetIter)) {
              OA_ptr<Alias::CallContext> cc = ccSetIter->current();
              
              //wrap stmt with cc
              Alias::StmtHwContext shwc(stmt,cc);
            
              // ======= Get AssignPairs for the Statement ========
              OA::OA_ptr<OA::AssignPairIterator> aPairIter;
              aPairIter = mIR->getAssignPairIterator(stmt);
              
              mStmtwCToDefs[shwc] = calcDefs(aPairIter,cc);
              aPairIter = mIR->getAssignPairIterator(stmt);
              mStmtwCToMustDefs[shwc] = calcMustDefs(aPairIter,cc);
              aPairIter = mIR->getAssignPairIterator(stmt);
              mStmtwCToUses[shwc] = calcUses(aPairIter,cc);
            }
        }
    }
  }

}



  } // end of namespace Activity
} // end of namespace OA
