/*! \file
  
  \brief Implementation of Activity::InterUseful

  \authors Michelle Strout, Barbara Kreaseck
  \version $Id: InterUseful.hpp,v 1.2 2005/06/10 02:32:02 mstrout Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/


#include <OpenAnalysis/Activity/InterUseful.hpp>
#include <map>

namespace OA {
  namespace Activity {

InterUseful::InterUseful() 
{ 
  mNumIter = 0;
  mHasCSResults = false;
}

bool
InterUseful::hasCSUseful()
{
  return mHasCSResults;
}

OA_ptr<Activity::UsefulStandard> 
InterUseful::getUsefulResults(ProcHandle proc)
{ 
  if (mProcToUsefulMap[proc].ptrEqual(0)) {
    OA_ptr<UsefulStandard> empty;
    empty = new UsefulStandard(proc);
    mProcToUsefulMap[proc] = empty;
  }
  return mProcToUsefulMap[proc];
}
  
OA_ptr<Activity::CSUseful> 
InterUseful::getCSUsefulResults(ProcHandle proc)
{ 
  assert(mHasCSResults);

  if (mProcToCSUsefulMap[proc].ptrEqual(0)) {
    OA_ptr<CSUseful> empty;
    empty = new CSUseful(proc);
    mProcToCSUsefulMap[proc] = empty;
  }
  return mProcToCSUsefulMap[proc];
}
  
int
InterUseful::getNumIter()
{ 
  return mNumIter; 
}

// *****************************************************************
// Output
// *****************************************************************

void 
InterUseful::dump(std::ostream& os, OA_ptr<IRHandlesIRInterface> ir)
{
  os << "====================== InterUseful" << std::endl;
  if (!mHasCSResults) {
    std::map<ProcHandle,OA_ptr<UsefulStandard> >::iterator mapIter;
    for (mapIter=mProcToUsefulMap.begin();
         mapIter!=mProcToUsefulMap.end();
         mapIter++) 
      {
        OA_ptr<UsefulStandard> results = mapIter->second;
        results->dump(os,ir);
        os << "-------------------------------" << std::endl;
      }
  } else {
    std::map<ProcHandle,OA_ptr<CSUseful> >::iterator mapIter;
    for (mapIter=mProcToCSUsefulMap.begin();
         mapIter!=mProcToCSUsefulMap.end();
         mapIter++) 
      {
        OA_ptr<CSUseful> results = mapIter->second;
        results->dump(os,ir);
        os << "-------------------------------" << std::endl;
      }
  }
   
  os << "\nInterUseful Number of Iterations = " << mNumIter
     << "\n\n====================== End InterUseful\n\n";
}

    
// *****************************************************************
// Annotation Interface
// *****************************************************************
void 
InterUseful::output(IRHandlesIRInterface &ir) const
{
  sOutBuild->objStart("InterUseful"); {
    if (!mHasCSResults) {
      sOutBuild->mapStart("mProcToUsefulMap","ProcHandle","UsefulStandard");
      std::map<ProcHandle,OA_ptr<UsefulStandard> >::const_iterator mapIter;
      for (mapIter=mProcToUsefulMap.begin();
           mapIter!=mProcToUsefulMap.end();
           mapIter++)
        {
          ProcHandle proc = mapIter->first;
          OA_ptr<UsefulStandard> results = mapIter->second;
          
          sOutBuild->mapEntryStart(); {
            sOutBuild->mapKeyStart(); {
              sOutBuild->outputIRHandle(proc,ir);
            } sOutBuild->mapKeyEnd();
            sOutBuild->mapValueStart(); {
              results->output(ir);
            } sOutBuild->mapValueEnd();
          } sOutBuild->mapEntryEnd();
        }
      sOutBuild->mapEnd("mProcToUsefulMap");
    }
    else {
      sOutBuild->mapStart("mProcToCSUsefulMap","ProcHandle","CSUseful");
      std::map<ProcHandle,OA_ptr<CSUseful> >::const_iterator mapIter;
      for (mapIter=mProcToCSUsefulMap.begin();
           mapIter!=mProcToCSUsefulMap.end();
           mapIter++)
        {
          ProcHandle proc = mapIter->first;
          OA_ptr<CSUseful> results = mapIter->second;
          
          sOutBuild->mapEntryStart(); {
            sOutBuild->mapKeyStart(); {
              sOutBuild->outputIRHandle(proc,ir);
            } sOutBuild->mapKeyEnd();
            sOutBuild->mapValueStart(); {
              results->output(ir);
            } sOutBuild->mapValueEnd();
          } sOutBuild->mapEntryEnd();
        }
      sOutBuild->mapEnd("mProcToCSUsefulMap");
    }

  } sOutBuild->objEnd("InterUseful");
  
}

// *****************************************************************
// Construction methods 
// *****************************************************************

void 
InterUseful::mapProcToUseful(ProcHandle proc, OA_ptr<UsefulStandard> useful)
{
  mProcToUsefulMap[proc] = useful;
}

void 
InterUseful::mapProcToCSUseful(ProcHandle proc, OA_ptr<CSUseful> useful)
{
  mProcToCSUsefulMap[proc] = useful;
  mHasCSResults = true;
}

void 
InterUseful::setNumIter(int n)
{ 
  mNumIter = n; 
}


  } // end of Activity namespace
} // end of OA namespace

