function [x_m y_m MatData MatTracks]=TracksToMatrix(xmin,ymin,nx,ny,posting,track_coord,track_values,theta,varargin),
%TRACKSTOMATRIX - Create a map from tracks
%
%   This routine creates a map with average values of tracks using the DACE toolbox.
%   xmin and ymin are two scalars are the extreme values of the data matrix
%   trac_coord is an exp file containing the coordinates of the tracks (x and y)
%   trav_values is a vector with the values along the track coordinates
%   posting is a scalar giving the posting of the matrix in meters
%   theta is a parameter of the correlation function
%
%   Usage:
%      [x_m y_m MatData]=TracksToMatrix(xmin,ymin,nx,ny,posting,track_coord,track_values),
%
%   Example:
%      [x_m y_m Thickness]=TracksToMatrix(-10^6,!0^6,200,180,1000,'trackcoord.exp',thickness_values),
%
%   See also: CREATEDATABOUNDARIES, CREATEDATAMATRIX

if nargin~=8 & nargin~=11,
	error(' TracksToMatrix error message: wrong number of arguments')
end

%First create the x_m and y_m fot the matrix
x_f=[xmin:posting:xmin+posting*nx]';
y_f=[ymin:posting:ymin+posting*ny]';

%Now create a bigger map we will then truncate
x_m=[xmin-posting*nx/2:posting:xmin+posting*nx*3/2]';
y_m=[ymin-posting*ny/2:posting:ymin+posting*ny*3/2]';

%Create DataMatrix with local averaging of tracks
[Mvalue Mx My]=CreateDataMatrix(x_m,y_m,track_coord,track_values);

MatTracks=Mvalue;
%Create vector for these coordinates and values
Svalue=sparse(Mvalue);
Sx=sparse(Mx);
Sy=sparse(My);
[i,j,Values]=find(Svalue);
[i,j,X]=find(Sx);
[i,j,Y]=find(Sy);
Coord=[X,Y];

%Create boundaries for the track if specified
if nargin==11,
	%Find out the elements to counstrain the border
	x_matrix=varargin{1};
	y_matrix=varargin{2};
	datamatrix=varargin{3};

	%Create values on the border
	[Xedge,Yedge,EdgeValues]=CreateDataBoundaries(x_m,y_m,x_matrix,y_matrix,datamatrix);

	%Add these values to the track values
	Values=[Values;EdgeValues];
	Coord=[X,Y;Xedge,Yedge];
end

%Create model for data
[dmodel,perf]=dacefit(Coord,Values,@regpoly1,@corrgauss,theta);

%Create design site(points where we are looking for the data)
Points=gridsamp([x_f(1)+posting/2,y_f(1)+posting/2;x_f(end)-posting/2,y_f(end)-posting/2],[length(x_f)-1;length(y_f)-1]);

%Compute data on these points
VecData=predictor(Points,dmodel);

%Reshape to get a matrix
MatData=reshape(VecData,ny,nx);
