/* $Header: /home/cvs/bp/oofem/sm/src/isodamagemodel.C,v 1.4.4.1 2004/04/05 15:19:47 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/

#include "trabbonematerial.h"
#include "gausspnt.h"
#include "flotmtrx.h"
#include "flotarry.h"
#include "structuralcrosssection.h"
#include "structuralmaterial.h"
#include "contextioerr.h"
#include "mathfem.h"

namespace oofem {

/////////////////////////////////////////////////////////////////
////////////////TRABECULAR BONE MATERIAL/////////////////////////
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: CONSTRUCTOR
//

TrabBoneMaterial :: TrabBoneMaterial (int n, Domain *d) : StructuralMaterial (n,d)
{
}

//
// END: CONSTRUCTOR
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: DESTRUCTOR
//

//
// END: DESTRUCTOR
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: MODE COMPATIBILITY
// returns whether receiver supports given mode

int
TrabBoneMaterial :: hasMaterialModeCapability (MaterialMode mode)
{
  if ((mode == _1dMat)) return 1;
  return 0;
}

//
// END: MODE COMPATIBILITY
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: SUBROUTINE OF ALPHA EVALUATION
//

void TrabBoneMaterial::computeCumPlastStrain (double& alpha, GaussPoint* gp, TimeStep* atTime)
{
  TrabBoneMaterialStatus *status = (TrabBoneMaterialStatus*) this -> giveStatus (gp);
  alpha = status -> giveTempAlpha();
}

//
// END: SUBROUTINE OF ALPHA EVALUATION
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: EVALUATION OF STIFFNESS MATRIX
//

void
TrabBoneMaterial::give1dStressStiffMtrx (FloatMatrix& answer,
                                   MatResponseForm form,MatResponseMode mode,GaussPoint* gp,
                                   TimeStep* atTime)
{
  TrabBoneMaterialStatus *status = (TrabBoneMaterialStatus*) this -> giveStatus (gp);

  double epsnew;
  double epsp,depsp;
  double dam,alpha;
  double matconstc;
  FloatArray epsnewArray;
 
  if (mode == ElasticStiffness)
  {
    answer.resize(1,1);
    answer.at(1,1)=E0;
  }
  else if (mode == SecantStiffness)
  {
    dam = status -> giveTempDam();
    matconstc = status -> giveMatConstC();

    answer.resize(1,1);
    answer.at(1,1)=(1.0-dam)*E0+matconstc;
  }
  else
  {
    epsnew = status -> giveTempStrainVector().at(1);
    epsnewArray = status -> giveTempStrainVector();
    epsp = status -> giveTempPlasStrainVector().at(1);
    depsp = status -> giveTempIncPlasStrainVector().at(1);
    alpha = status -> giveTempAlpha();
    dam = status -> giveTempDam();

    answer.resize(1,1);

    computeDensification(gp, epsnewArray);
    matconstc = status -> giveMatConstC();

    if (depsp != 0.0)
    {
      answer.at(1,1) = (1.0-dam)*(E0*(Eil+Ek))/(E0+Eil+Ek)
      -E0*E0*(epsnew-epsp)/(E0+Eil+Ek)*adam*exp(-adam*alpha)*depsp/fabs(depsp)+matconstc;
    }
    else
    {
      answer.at(1,1) = (1.0-dam)*E0+matconstc;
    }
  }
  status -> setSmtrx(answer.at(1,1));
}

//
// END: EVALUATION OF STIFFNESS MATRIX
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: SUBROUTINE OF PLASTIC RETURN ALGORITHM
//

void
TrabBoneMaterial :: performPlasticityReturn(GaussPoint* gp, const FloatArray& totalStrain)
{
  double epsnew,epsold;
  double epsp,depsp;
  double alpha;
  double sig,sigp,sigY;
  double gNewton,dgNewton;

  TrabBoneMaterialStatus *status = (TrabBoneMaterialStatus*) this -> giveStatus (gp);

  epsnew = totalStrain.at(1);
  epsold = status -> giveStrainVector().at(1);
  epsp = status -> givePlasStrainVector().at(1);
  alpha = status -> giveAlpha();
  sig = status -> giveStressVector().at(1);
  sigp = E0*epsnew-(E0+Ek)*epsp;

  if(sigp < 0.0)
    sigY = SigYn;
  else 
    sigY = SigYp;

  if(sigp/fabs(sigp)*sigp > sigY + Eil*alpha+Eie*(1-exp(-kie*alpha)))
    {
      depsp = epsnew-epsold;
      gNewton = sigp/fabs(sigp)*(E0*epsnew-(E0+Ek)*(epsp+depsp))-(sigY+Eil*(alpha+
        	sigp/fabs(sigp)*depsp)+Eie*(1-exp(-kie*(alpha+sigp/fabs(sigp)*depsp))));
      while (fabs(gNewton) > 10.e-7)
      {
        gNewton = sigp/fabs(sigp)*(E0*epsnew-(E0+Ek)*(epsp+depsp))-(sigY+Eil*(alpha+
		sigp/fabs(sigp)*depsp)+Eie*(1-exp(-kie*(alpha+sigp/fabs(sigp)*depsp))));
        dgNewton = -sigp/fabs(sigp)*((E0+Ek)+Eil+
		Eie*kie*exp(-kie*(alpha+sigp/fabs(sigp)*depsp)));
        depsp += -gNewton / dgNewton ;
      }
      epsp += depsp;
    }
  else
    {
      depsp = 0.0;
    }

  alpha += fabs(depsp);

  status -> setTempEpsp(epsp);
  status -> setTempDepsp(depsp);
  status -> setTempAlpha(alpha);

}

//
// END: SUBROUTINE OF PLASTIC RETURN ALGORITHM
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: SUBROUTINE OF DENSIFICATION
//

void
TrabBoneMaterial :: computeDensification(GaussPoint* gp, const FloatArray& totalStrain)
{
  double epsnew,sigc;
  double matconstc;

  TrabBoneMaterialStatus *status = (TrabBoneMaterialStatus*) this -> giveStatus (gp);

  epsnew = totalStrain.at(1);

  if(epsnew > EpsC)
  {
    sigc=0.0;
    matconstc=0.0;
  }
  else 
  {
    sigc=Cc*(epsnew-EpsC)+Cc2*(epsnew-EpsC)*(epsnew-EpsC)*(epsnew-EpsC)*(epsnew-EpsC)*(epsnew-EpsC)*(epsnew-EpsC)*(epsnew-EpsC);
    matconstc=Cc+7*Cc2*(epsnew-EpsC)*(epsnew-EpsC)*(epsnew-EpsC)*(epsnew-EpsC)*(epsnew-EpsC)*(epsnew-EpsC);
  }
  status -> setSigC(sigc);
  status -> setMatConstC(matconstc);

}

//
// END: SUBROUTINE OF DENSIFICATION
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: FUNCTION FOR DAMAGE PARAMETER
//

double  
TrabBoneMaterial::computeDamageParam (double alpha, GaussPoint* gp)  
{
  double dam;
  if (alpha > 0.)
  {
    dam = 1.0-exp(-adam*alpha);
//    dam = adam*alpha;
  }
  else
  { 
    dam = 0.0;
  }
  if (alpha < 0.)
  {
    _error("Alpha less than zero. Not possible");
  }
  return dam;
}

//
// END: FUNCTION FOR DAMAGE PARAMETER
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: FUNCTION FOR DAMAGE EVALUATION
//

double
TrabBoneMaterial::computeDamage (GaussPoint* gp,  TimeStep* atTime)
{
  double alpha;

  computeCumPlastStrain(alpha, gp, atTime);

  double dam = computeDamageParam (alpha, gp);

  return dam;

}

//
// END: FUNCTION FOR DAMAGE EVALUATION
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: SUBROUTINE FOR EVALUATION OF TOTAL STRESS
// returns real stress vector in 3d stress space of receiver according to 
// previous level of stress and current
// strain increment, the only way, how to correctly update gp records
//

void
TrabBoneMaterial :: giveRealStressVector (FloatArray& answer, MatResponseForm form, GaussPoint* gp, 
                                    const FloatArray& totalStrain, 
                                    TimeStep* atTime)
{
  double epsnew,epsp;
  double dam;
  double sig,sigc;
  double dt;

  TrabBoneMaterialStatus *status = (TrabBoneMaterialStatus*) this -> giveStatus (gp);

  this->initGpForNewStep(gp);

  performPlasticityReturn(gp, totalStrain);

  dt = atTime ->  giveTimeIncrement();

  epsnew = totalStrain.at(1);
  
  computeDensification(gp, totalStrain);
    
  dam = computeDamage(gp, atTime);

  epsp = status -> giveTempPlasStrainVector().at(1);
  sigc = status -> giveSigC();
  sig = (1-dam)*E0*(epsnew-epsp)+sigc;
  
  answer.resize(1);
  answer.at(1) = sig;
  status -> setTempDam(dam);
  status -> letTempStrainVectorBe (totalStrain);
  status -> letTempStressVectorBe( answer ) ;

}

//
// END: SUBROUTINE FOR EVALUATION OF TOTAL STRESS
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: PARAMETERS OF INPUT FILE
//

IRResultType
TrabBoneMaterial :: initializeFrom (InputRecord* ir)
{
 const char *__proc = "initializeFrom"; // Required by IR_GIVE_FIELD macro
 IRResultType result;                   // Required by IR_GIVE_FIELD macro

 /// Read material properties here

 IR_GIVE_FIELD (ir, E0, IFT_TrabBoneMaterial_E0, "e0"); // Macro
 IR_GIVE_FIELD (ir, Eil, IFT_TrabBoneMaterial_Eil, "eil"); // Macro
 IR_GIVE_FIELD (ir, Eie, IFT_TrabBoneMaterial_Eie, "eie"); // Macro
 IR_GIVE_FIELD (ir, kie, IFT_TrabBoneMaterial_kie, "kie"); // Macro
 IR_GIVE_FIELD (ir, Ek, IFT_TrabBoneMaterial_Ek, "ek"); // Macro
 IR_GIVE_FIELD (ir, Cc, IFT_TrabBoneMaterial_Cc, "cc"); // Macro
 IR_GIVE_FIELD (ir, Cc2, IFT_TrabBoneMaterial_Cc2, "cc2"); // Macro
 IR_GIVE_FIELD (ir, EpsC, IFT_TrabBoneMaterial_EpsC, "epsc"); // Macro
 IR_GIVE_FIELD (ir, SigYp, IFT_TrabBoneMaterial_SigYp, "sigyp"); // Macro
 IR_GIVE_FIELD (ir, SigYn, IFT_TrabBoneMaterial_SigYn, "sigyn"); // Macro
 IR_GIVE_FIELD (ir, adam, IFT_TrabBoneMaterial_adam, "adam"); // Macro

 return StructuralMaterial::initializeFrom (ir);
} 

//
// END: PARAMETERS OF INPUT FILE
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
//////////////////TRABECULAR BONE STATUS/////////////////////////
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: CONSTRUCTOR
// init state variables

TrabBoneMaterialStatus::TrabBoneMaterialStatus (int n, Domain*d, GaussPoint* g):StructuralMaterialStatus(n,d,g)
{

	alpha = 0.0;
	tempAlpha = 0.0;
	dam = 0.0;
	tempDam = 0.0;
	smtrx = 0.0;
	slope = 0.0;
	sigC = 0.0;
	matConstC = 0.0;
	epsp.resize(1);
	tempEpsp.resize(1);
	epsp.at(1) = 0.0;
	tempEpsp.at(1) = 0.0;
	tempDepsp.resize(1);
	tempDepsp.at(1) = 0.0; 
}

//
// END: CONSTRUCTOR
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: DESTRUCTOR
// 

TrabBoneMaterialStatus::~TrabBoneMaterialStatus () 
{
}

//
// END: DESTRUCTOR
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: DEFINITION OF "GIVE PARAMETER"
//

double
TrabBoneMaterialStatus :: giveAlpha()
{
  return alpha;
}

double
TrabBoneMaterialStatus :: giveTempAlpha()
{
  return tempAlpha;
}

double
TrabBoneMaterialStatus :: giveDam()
{
  return dam;
}

double
TrabBoneMaterialStatus :: giveTempDam()
{
  return tempDam;
}

double
TrabBoneMaterialStatus :: giveSmtrx()
{
  return smtrx;
}

double
TrabBoneMaterialStatus :: giveSlope()
{
  return slope;
}

double
TrabBoneMaterialStatus :: giveSigC()
{
  return sigC;
}

double
TrabBoneMaterialStatus :: giveMatConstC()
{
  return matConstC;
}

FloatArray
TrabBoneMaterialStatus :: givePlasStrainVector()
{
  return epsp;
}

FloatArray
TrabBoneMaterialStatus :: giveTempPlasStrainVector()
{
  return tempEpsp;
}

FloatArray
TrabBoneMaterialStatus :: giveTempIncPlasStrainVector()
{
  return tempDepsp;
}

//
// END: DEFINITION OF "GIVE PARAMETER"
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: OUTPUT
// print state to output stream

void 
TrabBoneMaterialStatus :: printOutputAt  (FILE *file, TimeStep* tStep)
{
  StructuralMaterialStatus :: printOutputAt (file, tStep);
  fprintf (file,"status { ");
  fprintf (file,"plastrains %f, alpha %f, dam %f, Slope %f, Smtrx %f ",this->tempEpsp.at(1),
		    this->tempAlpha, this->tempDam,this->slope, this->smtrx);
  fprintf (file,"}\n");
}

//
// END: OUTPUT
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: INITIALIZE TEMP VARIABLE (UPDATED DURING ITERATIONS)
// initialize temporary state variables according to equilibriated state vars

void 
TrabBoneMaterialStatus::initTempStatus ()
{
  StructuralMaterialStatus :: initTempStatus();
  this->tempAlpha = this->alpha;
  this->tempDam = this->dam;
  this->tempEpsp= this->epsp;
}

//
// END: INITIALIZE TEMP VARIABLE (UPDATED DURING ITERATIONS)
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: SETS VARIABLE EQUAL TO TEMP VARIABLE AT THE END OF THE STEP
// Called when equlibrium reached, set equilibriated vars according to temporary (working) ones.

void 
TrabBoneMaterialStatus::updateYourself(TimeStep* atTime)
{
  StructuralMaterialStatus::updateYourself(atTime);
  this->alpha = this->tempAlpha;
  this->dam = this->tempDam;
  this->epsp= this->tempEpsp;
}

//
// END: SETS VARIABLE EQUAL TO TEMP VARIABLE AT THE END OF THE STEP
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: INTERRUPT RESTART UTILITY - SAVE
//

contextIOResultType
TrabBoneMaterialStatus::saveContext (DataStream* stream, ContextMode mode, void *obj)
{
 contextIOResultType iores;

 // save parent class status
 if ((iores = StructuralMaterialStatus :: saveContext (stream, mode, obj)) != CIO_OK) THROW_CIOERR(iores);

 // write a raw data
 //if (fwrite(&kappa,sizeof(double),1,stream) != CIO_OK) THROW_CIOERR(CIO_IOERR);
 //if (fwrite(&damage,sizeof(double),1,stream)!= CIO_OK) THROW_CIOERR(CIO_IOERR);

 return CIO_OK;
}

//
// END: INTERRUPT RESTART UTILITY - SAVE
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: INTERRUPT RESTART UTILITY - RESTORE
//

contextIOResultType
TrabBoneMaterialStatus::restoreContext(DataStream* stream, ContextMode mode, void *obj)
{
 contextIOResultType iores;

 // read parent class status
 if ((iores = StructuralMaterialStatus :: restoreContext (stream,mode,obj)) != CIO_OK) THROW_CIOERR(iores);

 // read raw data 
 //if (fread (&kappa,sizeof(double),1,stream) != 1) THROW_CIOERR(CIO_IOERR);
 //if (fread (&damage,sizeof(double),1,stream) != 1) THROW_CIOERR(CIO_IOERR);

 return CIO_OK;
}

//
// END: INTERRUPT RESTART UTILITY - RESTORE
/////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////
// BEGIN: CREATE STATUS
//

MaterialStatus *TrabBoneMaterial::CreateStatus (GaussPoint* gp) const
{
  TrabBoneMaterialStatus *status =
    new  TrabBoneMaterialStatus (1, StructuralMaterial :: giveDomain(), gp) ;
  return status ;
}

//
// END: CREATE STATUS
/////////////////////////////////////////////////////////////////

} // end namespace oofem
