/* $Header: /home/cvs/bp/oofem/oofemlib/src/loadtime.h,v 1.9 2003/04/06 14:08:25 bp Exp $ */
/*
 *
 *                 #####    #####   ######  ######  ###   ###
 *               ##   ##  ##   ##  ##      ##      ## ### ##
 *              ##   ##  ##   ##  ####    ####    ##  #  ##
 *             ##   ##  ##   ##  ##      ##      ##     ##
 *            ##   ##  ##   ##  ##      ##      ##     ##
 *            #####    #####   ##      ######  ##     ##
 *
 *
 *             OOFEM : Object Oriented Finite Element Code
 *
 *               Copyright (C) 1993 - 2008   Borek Patzak
 *
 *
 *
 *       Czech Technical University, Faculty of Civil Engineering,
 *   Department of Structural Mechanics, 166 29 Prague, Czech Republic
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * The original idea for this class comes from
 * Dubois-Pelerin, Y.: "Object-Oriented  Finite Elements: Programming concepts and Implementation",
 * PhD Thesis, EPFL, Lausanne, 1992.
 */


//   ********************************
//   *** CLASS LOAD-TIME FUNCTION ***
//   ********************************


#ifndef loadtime_h
#define loadtime_h

#include "femcmpnn.h"
#include "domain.h"
#include "classtype.h"
#include "valuemodetype.h"

namespace oofem {

/**
 * Abstract base class representing load time function. Classes derived from Load class typically
 * describe load from spatial point of view. The purpose of introducing load time function is to express
 * variation of some components in time. Load time function typically belongs to domain and is
 * attribute of one or more loads. Generally load time function is real function of time (\f$y=f(t)\f$).
 */
class LoadTimeFunction : public FEMComponent
{
    /*
     * This abstract class is the superclass of the classes that implement
     * functions  y = f(t) , where t is the time. These function are used for
     * weighing the loads (see TJR Hughes, "The Finite Element Method", p 677).
     * A load-time function is attribute of the domain. It is usually also at-
     * tribute of one or more loads.
     * DESCRIPTION
     * The parameters which describe the function are defined by the subclasses
     * of this class.
     * TASK
     * Returning the value 'y' at any abscissa 't'.
     */

 protected:
  /**
     By default, the increment of receiver is computed as a difference between values evaluated at given solution step and in previous step.
     However, if the solution step is the first step, the difference is typically set to the total value of receiver at the first step.
     This is quite natural, as a loading with constant time function is expected to be applied at first step.
     In certain cases, this default behavior has to be changed. The initial value (set by default to zero) 
     allows to set initial value of receiver. This initial value is used only when the increment of receiver is evaluated at first step,
     when result is defined as value of receiver at given step minus the initial value.
     This allows to correctly handle temperature loading, that is specified with respect to some reference temperature.
     In this case, the initial value should be set to the reference temperature, allowing to obtain correct temperate increment in first step.
   */
  double initialValue;
 public:

    /**
     * Constructor. Creates load time function with given number, belonging to given domain.
     * @param n load time function number
     * @param d domain to which new object will belongs.
     */
    LoadTimeFunction(int i, Domain *d) : FEMComponent(i, d) { initialValue = 0.0;}
    /// Destructor
    virtual ~LoadTimeFunction()  { }

    // computations
    /**
     * Returns the value of load time function at given time. Abstract service.
     * Must be implemented by derived classes.
     * @param t time
     * @return load time function value
     */
    double     evaluate(TimeStep *atTime, ValueModeType mode);

    // definition of a function

    /**
     * Returns a newly allocated load time function, with type depending on parameter.
     * Creates new object for following classes ConstantFunction
     * otherwise calls directly CreateUsrDefLoadTimeFunctionOfType global function to allocate
     * new instance of load time function of given type.
     * @param aClass string with load time function ID  name
     * @return newly allocated load time function of required type.
     * @see CreateUsrDefLoadTimeFunctionOfType function.
     */
    LoadTimeFunction *ofType(char *);
    /// Returns classType id of receiver.
    classType   giveClassID() const { return LoadTimeFunctionClass; }
    /// Returns class name of the receiver.
    const char *giveClassName() const { return "LoadTimeFunction"; }
    /**
     * Initializes receiver acording to object description stored in input record.
     * Must be implemented in derived classes
     */
    IRResultType initializeFrom(InputRecord *ir);
    /** Setups the input record string of receiver
     *  @param str string to be filled by input record
     *  @param keyword print record keyword (default true)
     */
    virtual int giveInputRecordString(std :: string &str, bool keyword = true);


    /**
     * Returns the value of load time function at given time. Abstract service.
     * Must be implemented by derived classes.
     * @param t time
     * @return load time function value
     */
    virtual double     __at(double)            { return 0.; }
    /**
     * Returns the first time derivative of load time function at given time. Abstract service.
     * Must be implemented by derived classes.
     * @param t time
     * @return load time function value
     */
    virtual double    __derAt(double) { return 0.; }
    /**
     * Returns the second time derivative of load time function at given time. Abstract service.
     * Must be implemented by derived classes.
     * @param t time
     * @return load time function value
     */
    virtual double    __accelAt(double) { return 0.; }
};

} // end namespace oofem
#endif // loadtime_h
