function bool=isresultconsistent(md,solution)
%ISRESULTCONSISTENT: check that results are consistent
%
%   This routine is called after a computation and check
%   that the values t the penalties are consitent
%
%   Usage:
%      bool=IsModelSelfConsistent(model,solution)

%tolerance we use in litmus tests for the consistency of the model
tolerance=10^-2;

%some checks
if (nargin~=2 | nargout~=1)
	help isresultconsistent
	error(' ');
end

%initialize output as 1 -> success
bool=1;

%do not check results if qmu analysis
if md.qmu_analysis,
	return
end

%DIAGNOSTIC
if strcmp(solution,'diagnostic'),

	if strcmpi(md.type,'3d')
		fields1={'results.diagnostic.vx','results.diagnostic.vy','results.diagnostic.vz','results.diagnostic.vel','results.diagnostic.pressure'};
		fields2={'results.diagnostic.vel','results.diagnostic.pressure'};
	else
		fields1={'results.diagnostic.vx','results.diagnostic.vy','results.diagnostic.vel'};
		fields2={'results.diagnostic.vel'};
	end

	%check size
	if ~testsize(md,fields1,md.numberofgrids),
		bool=0; return
	end

	%no NAN
	if ~testnan(md,fields1),
		bool=0; return
	end

	%check value is real
	if ~testreal(md,fields1),
		bool=0; return
	end

	%check value>=0
	if ~testpositive(md,fields2),
		bool=0; return
	end

end

%CONTROL
if strcmp(solution,'control'),

	fields1={'results.control.vx','results.control.vy','results.control.vel','results.control.parameter'};
	fields2={'results.control.vel','results.control.J'};

	%check size
	if ~testsize(md,fields1,md.numberofgrids),
		bool=0; return
	end

	%no NAN
	if ~testnan(md,fields1),
		bool=0; return
	end

	%check value is real
	if ~testreal(md,fields1),
		bool=0; return
	end

	%check value>=0
	if ~testpositive(md,fields2),
		bool=0; return
	end

	%check inversed parameter
	if ~isnan(md.maxcontrolconstraint),
		if any(md.results.control.parameter>md.maxcontrolconstraint)
			disp(['''control'' result not consistent: inverse parameter is greater than ' num2str(md.maxcontrolconstraint)]);
			bool=0; return; 
		end
	end
	if ~isnan(md.mincontrolconstraint),
		if any(md.results.control.parameter<md.mincontrolconstraint)
			disp(['''control'' result not consistent: inverse parameter is smaller than ' num2str(md.mincontrolconstraint)]);
			bool=0; return; 
		end
	end

end

%THERMAL
if strcmp(solution,'thermal')

	for iter=1:length(md.results.thermal)

		fields1={['results.thermal(' num2str(iter) ').temperature'],['results.thermal(' num2str(iter) ').melting']};
		fields2={['results.thermal(' num2str(iter) ').temperature']};

		%check size
		if ~testsize(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%no NAN
		if ~testnan(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%check value is real
		if ~testreal(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%check value>=0
		if ~testpositive(md,fields2,md.numberofgrids),
			bool=0; return
		end

		%check melting (<=0 via penalties)
		if any(abs(md.results.thermal(iter).melting(md.numberofgrids2d+1:end))>tolerance)
			disp(['''thermal'' result not consistent: increase penalty_melting (negative melting)']);
			bool=0; return; 
		end
	end

end

if strcmp(solution,'transient') 

	for iter=1:length(md.results.transient)

		if strcmpi(md.type,'3d'),
			fields1={['results.transient(' num2str(iter) ').vx'],['results.transient(' num2str(iter) ').vy'],...
				['results.transient(' num2str(iter) ').vz'],['results.transient(' num2str(iter) ').vel'],...
				['results.transient(' num2str(iter) ').bed'],['results.transient(' num2str(iter) ').surface'],...
				['results.transient(' num2str(iter) ').thickness'],...
				['results.transient(' num2str(iter) ').temperature'],['results.transient(' num2str(iter) ').melting']};
			fields2={['results.transient(' num2str(iter) ').vel'],['results.transient(' num2str(iter) ').thickness'],...
				['results.transient(' num2str(iter) ').temperature']};
		else
			fields1={['results.transient(' num2str(iter) ').vx'],['results.transient(' num2str(iter) ').vy'],...
				['results.transient(' num2str(iter) ').vel'],...
				['results.transient(' num2str(iter) ').bed'],['results.transient(' num2str(iter) ').surface'],...
				['results.transient(' num2str(iter) ').thickness']};
			fields2={['results.transient(' num2str(iter) ').vel'],['results.transient(' num2str(iter) ').thickness']};
		end

		%check size
		if ~testsize(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%no NAN
		if ~testnan(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%check value is real
		if ~testreal(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%check value>=0
		if ~testpositive(md,fields2,md.numberofgrids),
			bool=0; return
		end

		%check melting (<=0 via penalties)
		if strcmpi(md.type,'3d'),
			if any(abs(md.results.transient(iter).melting(md.numberofgrids2d+1:end))>tolerance)
				disp(['''thermal'' result not consistent: increase penalty_melting (negative melting)']);
				bool=0; return; 
			end
		end
	end
end
end % end function

function bool=testsize(md,fields,fieldsize)
	%TESTSIZE - test size of a field
	bool=1;
	for i=1:length(fields),
		if length(eval(['md.' fields{i}]))~=fieldsize
			disp(['results not consistent: field ' fields{i} ' size should be ' num2str(md.numberofgrids)]);
			bool=0; return;
		end
	end
end

function bool=testnan(md,fields,nanfield)
	%TESTNAN - test nan values of a field
	bool=1;
	for i=1:length(fields),
		if any(isnan(eval(['md.' fields{i}]))),
			disp(['results not consistent: NaN values in field ' fields{i}]);
			bool=0; return;
		end
	end
end

function bool=testreal(md,fields,realfield)
	%TESTREAL - test real values of a field
	bool=1;
	for i=1:length(fields),
		if any(eval(['~isreal(md.' fields{i} ')'])),
			disp(['results not consistent: complex values in field ' fields{i}]);
			bool=0; return;
		end
	end
end

function bool=testpositive(md,fields,positivefield)
	%TESTPOSITIVE - test positive values of a field
	bool=1;
	for i=1:length(fields),
		if any(eval(['md.' fields{i} '<0'])),
			disp(['results not consistent: negative values in field ' fields{i}]);
			bool=0; return;
		end
	end
end
