/*\file Shp2Kml.c
 *\brief: shp to kml file conversion mex module.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./Shp2Kml.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	int i,verbose=1;

	/*input: */
	char    *filshp=NULL,*filkml=NULL;
	int     sgn;

	Options* options=NULL;
	double   cm=0.,sp=0.;

	/* output: */
	int     iret=0;

	#ifndef _HAVE_SHAPELIB_ //only works if shapelib library has been compiled in.
	_error_(" Shapelib not available! Cannot carry out shp file translation!");
	#endif

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if (nlhs > NLHS) {
		Shp2KmlUsage();
		_error_("Shp2Kml usage error");
	}
	if (nrhs < NRHS) {
		Shp2KmlUsage();
		_error_("Shp2Kml usage error");
	}

	/*Input datasets: */
	if (verbose) printf("Fetching inputs:\n");
	FetchMatlabData(&filshp,SHP_IN);
	if (verbose) printf("  filshp=\"%s\"\n",filshp);
	FetchMatlabData(&filkml,KML_IN);
	if (verbose) printf("  filkml=\"%s\"\n",filkml);
	FetchMatlabData(&sgn,SGN_IN);
	if (verbose) printf("  sgn=%d\n",sgn);

	if (verbose) printf("Parsing options:\n");
	options=new Options(NRHS,nrhs,prhs);
	if (options->Size()) for(i=0;i<options->Size();i++) ((Option*)options->GetObjectByOffset(i))->DeepEcho();
	/*  defaults are in Xy2lldef, so don't duplicate them here, and only use user values if both have been specified  */
	if (options->GetOption("central_meridian") || options->GetOption("standard_parallel")) {
		options->Get(&cm,"central_meridian");
		if (verbose) printf("  cm=%g\n",cm);
		options->Get(&sp,"standard_parallel");
		if (verbose) printf("  sp=%g\n",sp);
	}

	/*some checks*/
	if (verbose) printf("Checking inputs:\n");

//	if (sgn != +1 && sgn != -1) _error_("Hemisphere sgn=%d must be +1 (north) or -1 (south).",sgn);
	if (fabs(cm)      > 180.) _error_("Central meridian cm=%g must be between -180 (west) and +180 (east) degrees.",cm);
	if (sp < 0. || sp >  90.) _error_("Standard parallel sp=%g must be between 0 and 90 degrees (in specified hemisphere).",sp);

	/* Run core computations: */
	if (verbose) printf("Calling core:\n");
	if (options->GetOption("central_meridian") && options->GetOption("standard_parallel"))
		iret=Shp2Kmlx(filshp,filkml,
					  sgn,cm,sp);
	else
		iret=Shp2Kmlx(filshp,filkml,
					  sgn);
	if (verbose) printf("  iret=%d\n",iret);

	/*Write data: */
	WriteMatlabData(RET_OUT,iret);

	/*Clean-up*/
	delete options;
	xfree((void**)&filkml);
	xfree((void**)&filshp);

	/*end module: */
	MODULEEND();
}

void Shp2KmlUsage(void)
{
	_printf_(true,"Shp2Kml - shp to kml file conversion module:\n");
	_printf_(true,"\n");
	_printf_(true,"   This module converts a file from shp to kml format.\n");
	_printf_(true,"\n");
	_printf_(true,"   Usage:\n");
	_printf_(true,"      [ret]=Shp2Kml(filshp,filkml,sgn,'param name',param,...);\n");
	_printf_(true,"\n");
	_printf_(true,"      filshp      file name of shp file to be read (char, extension optional)\n");
	_printf_(true,"      filkml      file name of kml file to be written (char)\n");
	_printf_(true,"      sgn         sign for hemisphere (double, +1 (north); -1 (south); 0 (no translation))\n");
	_printf_(true,"\n");
	_printf_(true,"      central_meridian     central meridian (double, optional, but must specify with sp)\n");
	_printf_(true,"      standard_parallel    standard parallel (double, optional, but must specify with cm)\n");
	_printf_(true,"\n");
	_printf_(true,"      ret         return code (non-zero for warning)\n");
	_printf_(true,"\n");
	_printf_(true,"   Examples:\n");
	_printf_(true,"      [ret]=Shp2Kml('file.shp','file.kml', 1);\n");
	_printf_(true,"      [ret]=Shp2Kml('file.shp','file.kml', 1,'central_meridian',45,'standard_parallel',70,'holes');\n");
	_printf_(true,"      [ret]=Shp2Kml('file.shp','file.kml',-1,'central_meridian', 0,'standard_parallel',71,'holes');\n");
	_printf_(true,"\n");
}

