/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "control_core"

#include "./parallel.h"
#include "../issm.h"

void control_core(DataSet* results,FemModel* fems, ParameterInputs* inputs){

	extern int my_rank;

	/*fem models: */
	FemModel* fem_dh=NULL;

	/*output: */
	Result* result=NULL;

	/*Intermediary: */
	Vec     u_g=NULL;
	double  search_scalar;
	char*   control_type=NULL;
	double* fit=NULL;
	double* optscal=NULL;
	double* u_g_obs=NULL;
	int     gsize;
	double* maxiter=NULL;
	double  tolx;
	double* param_g=NULL;
	Vec     grad_g=NULL;
	Vec     new_grad_g=NULL;
	Vec     grad_g_old=NULL;
	double* grad_g_double=NULL;
	double  mincontrolconstraint;
	double  maxcontrolconstraint;
	int     nsteps,n,i;
	double* J=NULL;
	OptArgs optargs;
	OptPars optpars;
	Param*  param=NULL;

	/*flags: */
	int   analysis_type;
	int   sub_analysis_type;
	int debug=0;
	int dim=-1;
	int ishutter=0;
	int ismacayealpattyn=0;
	int isstokes=0;
	int numberofdofspernode_sl;
	int numberofdofspernode_dh;
	int numberofdofspernode_ds;
	int numberofnodes;

	/*recover fem models: */
	fem_dh=fems+0;

	//first recover parameters common to all solutions
	fem_dh->parameters->FindParam((void*)&debug,"debug");
	fem_dh->parameters->FindParam((void*)&dim,"dim");
	fem_dh->parameters->FindParam((void*)&ishutter,"ishutter");
	fem_dh->parameters->FindParam((void*)&ismacayealpattyn,"ismacayealpattyn");
	fem_dh->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	fem_dh->parameters->FindParam((void*)&isstokes,"isstokes");

	/*Recover parameters used throughout the solution:*/
	fem_dh->parameters->FindParam((void*)&nsteps,"nsteps");
	fem_dh->parameters->FindParam((void*)&control_type,"control_type");
	fem_dh->parameters->FindParam((void*)&fit,"fit");
	fem_dh->parameters->FindParam((void*)&optscal,"optscal");
	fem_dh->parameters->FindParam((void*)&maxiter,"maxiter");
	fem_dh->parameters->FindParam((void*)&tolx,"tolx");
	fem_dh->parameters->FindParam((void*)&mincontrolconstraint,"mincontrolconstraint");
	fem_dh->parameters->FindParam((void*)&maxcontrolconstraint,"maxcontrolconstraint");
	fem_dh->parameters->FindParam((void*)&param_g,"param_g");
	fem_dh->parameters->FindParam((void*)&u_g_obs,"u_g_obs");
	fem_dh->parameters->FindParam((void*)&analysis_type,"analysis_type");
	fem_dh->parameters->FindParam((void*)&sub_analysis_type,"sub_analysis_type");
	fem_dh->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	gsize=fem_dh->nodes->NumberOfDofs();

	/*Initialize misfit: */
	J=(double*)xmalloc(nsteps*sizeof(double));

	/*erase useless parameters: */
	param=(Param*)fem_dh->parameters->FindParamObject("param_g");
	fem_dh->parameters->DeleteObject((Object*)param);

	param=(Param*)fem_dh->parameters->FindParamObject("u_g_obs");
	fem_dh->parameters->DeleteObject((Object*)param);

	param=(Param*)fem_dh->parameters->FindParamObject("u_g");
	fem_dh->parameters->DeleteObject((Object*)param);

	/*Start looping: */
	for(n=0;n<nsteps;n++){

		_printf_("\n%s%i%s%i\n","   control method step ",n+1,"/",nsteps);
		inputs->Add(control_type,param_g,2,numberofnodes);
		inputs->Add("fit",fit[n]);

		/*Update parameters: */
		UpdateFromInputsx(fem_dh->elements,fem_dh->nodes,fem_dh->loads, fem_dh->materials,inputs);

		_printf_("%s\n","      computing gradJ...");
		grad_g=GradJCompute(inputs,fem_dh,u_g_obs);
		_printf_("%s\n","      done.");

		_printf_("%s\n","      normalizing directions...");
		Orthx(&new_grad_g,grad_g,grad_g_old);
		VecFree(&grad_g); grad_g=new_grad_g;
		VecFree(&grad_g_old); grad_g_old=grad_g;
		VecToMPISerial(&grad_g_double,grad_g);
		_printf_("%s\n","      done.");

		_printf_("%s\n","      optimizing along gradient direction...");
		optargs.femmodel=fem_dh; optargs.param_g=param_g; optargs.u_g_obs=u_g_obs; optargs.grad_g=grad_g_double; optargs.inputs=inputs;optargs.n=n;
		optpars.xmin=0; optpars.xmax=1; optpars.tolerance=tolx; optpars.maxiter=(int)maxiter[n];
		BrentSearch(&search_scalar,J+n,&optpars,&objectivefunctionC,&optargs);
		_printf_("%s\n","      done.");

		_printf_("%s\n","      updating parameter using optimized search scalar...");
		for(i=0;i<gsize;i++)param_g[i]=param_g[i]+search_scalar*optscal[n]*grad_g_double[i];
		_printf_("%s\n","      done.");

		_printf_("%s\n","      constraining the new distribution...");    
		ControlConstrainx( param_g,gsize,mincontrolconstraint,maxcontrolconstraint,control_type);
		_printf_("%s\n","      done.");

		_printf_("%s%i%s%g\n","      value of misfit J after optimization #",n,": ",J[n]);

		/*some freeing:*/
		xfree((void**)&grad_g_double);
	}

	/*Write results to disk: */
	_printf_("%s\n","      preparing final velocity solution...");
	inputs->Add(control_type,param_g,2,numberofnodes);
	inputs->Add("fit",fit[n]);

	/*Update parameters: */
	UpdateFromInputsx(fem_dh->elements,fem_dh->nodes,fem_dh->loads, fem_dh->materials,inputs);

	diagnostic_core_nonlinear(&u_g,NULL,NULL,fem_dh,inputs,analysis_type,sub_analysis_type);

	/*Plug results into output dataset: */
	result=new Result(results->Size()+1,0,1,"u_g",u_g);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"param_g",param_g,gsize);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"J",J,nsteps);
	results->AddObject(result);
}
