/*!\file:  control.cpp
 * \brief: control solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "control"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

int main(int argc,char* *argv){

	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;
	int   qmu_analysis=0;

	/*Fem models : */
	FemModel femmodels[1];

	/*Results: */
	DataSet* results=NULL;

	ParameterInputs* inputs=NULL;
	int waitonlock=0;

	/*Intermediary: */
	double* u_g_initial=NULL;
	Param*  param=NULL;


	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	_printf_("read and create finite element model:\n");
	_printf_("\n   reading diagnostic horiz model data:\n");
	CreateFemModel(&femmodels[0],fid,"control","");

	_printf_("initialize inputs:\n");
	femmodels[0].parameters->FindParam((void*)&u_g_initial,"u_g");
	femmodels[0].parameters->FindParam((void*)&numberofnodes,"numberofnodes");

	inputs=new ParameterInputs;
	inputs->Add("velocity",u_g_initial,3,numberofnodes);

	/*erase velocities: */
	param=(Param*)femmodels[0].parameters->FindParamObject("u_g");
	femmodels[0].parameters->DeleteObject((Object*)param);

	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());

	/*are we running the solution sequence, or a qmu wrapper around it? : */
	femmodels[0].parameters->FindParam((void*)&qmu_analysis,"qmu_analysis");
	if(!qmu_analysis){

		/*run control analysis: */
		_printf_("call computational core:\n");
		control_core(results,femmodels,inputs);

	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on control core:\n");

#ifdef _HAVE_DAKOTA_ 
		Qmux(&femmodels[0],inputs,DiagnosticAnalysisEnum(),NoneAnalysisEnum());
#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
#endif
	}

	_printf_("process results:\n");
	ProcessResults(&results,&femmodels[0],ControlAnalysisEnum());

	_printf_("write results to disk:\n");
	OutputResults(results,outputfilename);

	_printf_("write lock file:\n");
	femmodels[0].parameters->FindParam((void*)&waitonlock,"waitonlock");
	if (waitonlock){
		WriteLockFile(lockname);
	}

	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 

	/*end module: */
	MODULEEND();
	
	return 0; //unix success return;
}
