/*!\file GroundinglineMigrationx
 * \brief: migration grounding line position.
 */

#include "../../shared/shared.h"
#include "../../io/io.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../Container/Container.h"
#include "./GroundinglineMigrationx.h"

void GroundinglineMigrationx(Elements* elements,Nodes* nodes, Vertices* vertices,Loads* loads,Materials* materials, Parameters* parameters){

	int      i, migration_style;
	double*  vertices_potentially_ungrounding = NULL;
	double*  vertices_ungrounding             = NULL;
	Element* element                          = NULL;
	
	_printf_(VerboseModule(),"   Migrating grounding line\n");
	
	/*retrieve parameters: */
	parameters->FindParam(&migration_style,GroundinglineMigrationEnum);
	if(migration_style==NoneEnum) return;
	if(migration_style!=AgressiveMigrationEnum && migration_style!=SoftMigrationEnum) _error_("%s not supported yet!",EnumToStringx(migration_style));

	if(migration_style==SoftMigrationEnum){
		/*Create flag for grounded vertices above the hydrostatic equilibrium: */
		vertices_potentially_ungrounding=PotentialSheetUngrounding(elements,vertices,parameters);

		/*propagate ice shelf into connex areas of the ice sheet that potentially want to unground: */
		vertices_ungrounding=PropagateFloatingiceToGrounded(elements,nodes,vertices,parameters,vertices_potentially_ungrounding);
	}

	/*Migrate grounding line : */
	for(i=0;i<elements->Size();i++){
		element=(Element*)elements->GetObjectByOffset(i);
		element->MigrateGroundingLine(vertices_ungrounding);
	}

	/*Synchronise mask: */
	for(i=0;i<elements->Size();i++){
		element=(Element*)elements->GetObjectByOffset(i);
		element->ShelfSync();
	}

	/*free ressouces: */
	xfree((void**)&vertices_potentially_ungrounding);
	xfree((void**)&vertices_ungrounding);
}
/*FUNCTION PotentialSheetUngrounding {{{1*/
double*    PotentialSheetUngrounding(Elements* elements,Vertices* vertices,Parameters* parameters){ 

	int      i,numberofvertices;
	double*  vertices_potentially_ungrounding      = NULL;
	Vec      vec_vertices_potentially_ungrounding  = NULL;
	Element* element                               = NULL;

	/*Initialize vector with number of vertices*/
	numberofvertices=vertices->NumberOfVertices();
	vec_vertices_potentially_ungrounding=NewVec(numberofvertices); //grounded vertex that could start floating

	/*Fill vector vertices_potentially_floating: */
	for(i=0;i<elements->Size();i++){
		element=(Element*)elements->GetObjectByOffset(i);
		element->PotentialSheetUngrounding(vec_vertices_potentially_ungrounding);
	}

	/*Assemble vector: */
	VecAssemblyBegin(vec_vertices_potentially_ungrounding);
	VecAssemblyEnd(vec_vertices_potentially_ungrounding);

	/*Serialize vector: */
	VecToMPISerial(&vertices_potentially_ungrounding,vec_vertices_potentially_ungrounding);

	/*free ressouces and return: */
	VecFree(&vec_vertices_potentially_ungrounding);
	return vertices_potentially_ungrounding;
}
/*}}}*/
/*FUNCTION PropagateFloatingiceToGrounded {{{1*/
double*    PropagateFloatingiceToGrounded(Elements* elements,Nodes* nodes,Vertices* vertices,Parameters* parameters,double* vertices_potentially_ungrounding){ 

	int      i,analysis_type;
	int      numberofvertices;
	int      nflipped,local_nflipped;
	double*  nodes_on_floatingice                  = NULL;
	double*  elements_neighbouring_floatingce      = NULL;
	Vec      vec_elements_neighbouring_floatingice = NULL;
	Vec      vec_nodes_on_floatingice              = NULL;
	Node*    node                                  = NULL;
	Element* element                               = NULL;


	/*recover parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	numberofvertices=vertices->NumberOfVertices();

	/*recover vec_nodes_on_floatingice*/
	vec_nodes_on_floatingice=NewVec(numberofvertices);

	/*Loop through nodes, and fill nodes_on_floatingice: */
	for(i=0;i<nodes->Size();i++){
		node=(Node*)nodes->GetObjectByOffset(i);
		if(node->InAnalysis(analysis_type)){
			if(node->IsFloating()){
				VecSetValue(vec_nodes_on_floatingice,node->Sid(),1.0,INSERT_VALUES);
			}
		}
	}

	/*Assemble vector and serialize: */
	VecAssemblyBegin(vec_nodes_on_floatingice);
	VecAssemblyEnd(vec_nodes_on_floatingice);
	VecToMPISerial(&nodes_on_floatingice,vec_nodes_on_floatingice);

	nflipped=1; //bootstrap
	while(nflipped){
		
		/*Vector of size number of elements*/
		vec_elements_neighbouring_floatingice=NewVec(elements->NumberOfElements(),true);

		/*Figure out if any of the nodes of the element will be floating -> elements neighbouting the floating ice*/
		for(i=0;i<elements->Size();i++){
			element=(Element*)elements->GetObjectByOffset(i);
			VecSetValue(vec_elements_neighbouring_floatingice,element->Sid(),element->IsNodeOnShelfFromFlags(nodes_on_floatingice)?1.0:0.0,INSERT_VALUES);
		}

		/*Assemble vector and serialize: */
		VecAssemblyBegin(vec_elements_neighbouring_floatingice);
		VecAssemblyEnd(vec_elements_neighbouring_floatingice);
		VecToMPISerial(&elements_neighbouring_floatingce,vec_elements_neighbouring_floatingice);

		/*Go through elements_neighbouring_floatingce, and update vector of the nodes that will start floating*/
		local_nflipped=0;
		for(i=0;i<elements->Size();i++){
			element=(Element*)elements->GetObjectByOffset(i);
			if(elements_neighbouring_floatingce[element->Sid()]){
				local_nflipped+=element->UpdatePotentialSheetUngrounding(vertices_potentially_ungrounding,vec_nodes_on_floatingice,nodes_on_floatingice);
			}
		}
		VecAssemblyBegin(vec_nodes_on_floatingice);
		VecAssemblyEnd(vec_nodes_on_floatingice);
		
		MPI_Allreduce(&local_nflipped,&nflipped,1,MPI_INT,MPI_SUM,MPI_COMM_WORLD);
		_printf_(VerboseConvergence(),"   number of grounded vertices  connected to grounding line: %i\n",nflipped);

		/*Avoid leaks: */
		xfree((void**)&elements_neighbouring_floatingce);
		xfree((void**)&nodes_on_floatingice);

		/*Assemble and serialize:*/
		VecFree(&vec_elements_neighbouring_floatingice);
		VecToMPISerial(&nodes_on_floatingice,vec_nodes_on_floatingice); 
	}

	/*Free ressources:*/
	VecFree(&vec_nodes_on_floatingice);
	xfree((void**)&elements_neighbouring_floatingce);

	return nodes_on_floatingice;
}
/*}}}*/
