function [elements,grids,loads,constraints,materials,part,tpart]=ModelProcessorSlopeCompute(md,solutiontype);
%MODELPROCESSORSLOPECOMPUTE - process model for a slope finite element computation
%
%   This routine uses all the informations in the model md and put them
%   into different structures (grids, elements, loads, constrained,materials)
%   that will be used to create the stiffness matrix and load vector.
%   After this routine, the model md should not be called until the end of the
%   solution sequence.
%
%   Usage:
%      [elements,grids,loads,constraints,materials,part,tpart]=ModelProcessorSlopeCompute(md,solutiontype)

global cluster

if cluster,
	%We are running in parallel, we need to partition the elements 
	element_partitioning=MeshPartition(md,numlabs);
else
	%We are running in serial, all elements belong to the same partition.
	element_partitioning=ones(md.numberofelements,1);
	labindex=1; %older versions of matlab do not include the parallel toolbox labindex variable.
end

%Allocate grids and elements
elements=struct('element',cell(md.numberofelements,1));
materials=struct('material',cell(0));
mygrids=zeros(md.numberofgrids,1); %this array determines grid partitioning.

%Build elements
if strcmpi(md.type,'2d'),
	pos=find(element_partitioning==labindex);
	[elements(pos).element]=deal(triaelem);

	elements(pos)=SetStructureField(elements(pos),'element','type','triaelem');
	elements(pos)=SetStructureField(elements(pos),'element','id',pos);
	elements(pos)=SetStructureField(elements(pos),'element','g',md.elements(pos,:));
	elements(pos)=SetStructureField(elements(pos),'element','b',md.bed(md.elements(pos,1:3)));
	elements(pos)=SetStructureField(elements(pos),'element','s',md.surface(md.elements(pos,1:3)));

else
	pos=find(element_partitioning==labindex);
	[elements(pos).element]=deal(pentaelem);

	elements(pos)=SetStructureField(elements(pos),'element','type','pentaelem');
	elements(pos)=SetStructureField(elements(pos),'element','id',pos);
	elements(pos)=SetStructureField(elements(pos),'element','g',md.elements(pos,1:6));
	elements(pos)=SetStructureField(elements(pos),'element','h',md.thickness(md.elements(pos,1:6)));
	elements(pos)=SetStructureField(elements(pos),'element','s',md.surface(md.elements(pos,1:6)));
	elements(pos)=SetStructureField(elements(pos),'element','b',md.bed(md.elements(pos,1:6)));
	elements(pos)=SetStructureField(elements(pos),'element','shelf',md.elementoniceshelf(pos));
	elements(pos)=SetStructureField(elements(pos),'element','onbed',md.elementonbed(pos));
	elements(pos)=SetStructureField(elements(pos),'element','onsurface',md.elementonsurface(pos));
	elements(pos)=SetStructureField(elements(pos),'element','collapse',1);
	elements(pos)=SetStructureField(elements(pos),'element','matid',pos);
	elements(pos)=SetStructureField(elements(pos),'element','melting',md.melting(md.elements(pos,1:6))/md.yts);
	elements(pos)=SetStructureField(elements(pos),'element','accumulation',md.accumulation(md.elements(pos,1:6))/md.yts);

end

if cluster, 
	%For elements, the corresponding grids belong to this cpu. Keep track of it. 
	mygrids(md.elements(el3pos,:))=1;
	mygrids(md.elements(el6pos,:))=1;
	
	%Figure out which grids from the partitioning belong to different element partitions. We'll 
	%call them 'border' grids.
	bordergrids=double(gplus(mygrids)>1);
else
	bordergrids=zeros(md.numberofgrids,1); %no partitioning serially.
end

%Get the grids set up:
grids=struct('grid',cell(md.numberofgrids,1));

pos=[1:md.numberofgrids]';
[grids(pos).grid]=deal(node);
grids(pos)=SetStructureField(grids(pos),'grid','id',pos);
grids(pos)=SetStructureField(grids(pos),'grid','x',md.x(pos));
grids(pos)=SetStructureField(grids(pos),'grid','y',md.y(pos));
grids(pos)=SetStructureField(grids(pos),'grid','z',md.z(pos));
grids(pos)=SetStructureField(grids(pos),'grid','s',(md.z(pos)-md.bed(pos))./md.thickness(pos));
grids(pos)=SetStructureField(grids(pos),'grid','onbed',md.gridonbed(pos));
grids(pos)=SetStructureField(grids(pos),'grid','border',bordergrids(pos));

%spc degrees of freedom:
%	 for each grid, 6 degrees of freedom are allowed. These dofs are numbered from 1 to 6. The first 3
%    deal with the (x,y,z) velocities, or deformations. The last 3 deal with the (x,y,z) rotations.
%    If a certain degree of freedom i (1<=i<=6) is constrained to the value 0, the number i should be added to the
%    gridset field of a grid.
%    The gridset field holds all the numbers corresponding to the dofs that have been constrained to 0 value. Because
%    we do not know firshand how many dofs have been constrained for a certain grid, we need a flexible way
%    of keeping track of these constraints. Hence gridset is a string array, of no given size, with no given
%    numerical order.
%    Ex: if a grid has 0 values for the x and z deformations, and 0 values for the y rotation, we could add any of the
%    following strings to the gridset: '135', '153', '315', etc ...
grids(pos)=SetStructureField(grids(pos),'grid','gridset','123456');

%for bed elements, free up the 3'rd dof for the first 3 grids 
if strcmpi(md.type,'3d'),
	for n=1:length(elements),
		%all dofs associated to this element are frozen, except if the element is on the bed, and acts as a 'fake' pentaelem, 
		%and a true 'macayeal' tria element.
		if strcmpi(solutiontype,'bed_slope_compute'),
			if elements(n).element.onbed,
				for j=1:3,
					grids(elements(n).element.g(j)).grid.gridset='23456';
				end
			end
		elseif strcmpi(solutiontype,'surface_slope_compute'),
			if elements(n).element.onsurface,
				for j=4:6,
					grids(elements(n).element.g(j)).grid.gridset='23456';
				end
			end
		end
	end
else
	for n=1:md.numberofgrids,
		grids(n).grid.gridset='23456';
	end
end
			

%Boundary conditions: no penalties to apply here
	
loads=struct('load',cell([0,1]));

%Single point constraints: none;
constraints=struct('constraint',cell(0,0));

%Last thing, return a partitioning vector, and its transpose.
[part,tpart]=PartitioningVector(md,grids);

end %end function
