/* \file WriteMatlabData.c:
 * \brief: general interface for writing data
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../include/include.h"
#include "../../shared/shared.h"

#ifdef _SERIAL_
#include <mex.h>

/*FUNCTION WriteMatlabData(mxArray** pdataref,DataSet* dataset){{{1*/
void WriteMatlabData(mxArray** pdataref,DataSet* dataset){

	mxArray* dataref=NULL;
	char* marshalled_dataset=NULL;
	int   marshalled_dataset_size;

	/*Write a dataset: */
	if(dataset){
			/* marshall the dataset: */
			marshalled_dataset=dataset->Marshall();
			marshalled_dataset_size=dataset->MarshallSize();
			
			dataref = mxCreateDoubleMatrix(0,0,mxREAL);
			mxSetM(dataref,(mwSize)(marshalled_dataset_size/sizeof(double)));
			mxSetN(dataref,(mwSize)1);
			mxSetPr(dataref,(double*)marshalled_dataset);	
	}
	else{
		/* return empty matrix: */
		dataref=mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
	
}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,Mat matrix){{{1*/
void WriteMatlabData(mxArray** pdataref,Mat matrix){
		
	mxArray* dataref=NULL;
	
	if(matrix){
		
		/*call toolkit routine: */
		PetscMatrixToMatlabMatrix(&dataref,matrix);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,double* matrix, int M,int N){{{1*/
void WriteMatlabData(mxArray** pdataref,double* matrix, int M,int N){
	
	mxArray* dataref=NULL;
	mxArray* tdataref=NULL;
		
	if(matrix){
		
		/*data is a double* pointer. Copy into a matrix: */
		tdataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(tdataref,(mwSize)N);
		mxSetN(tdataref,(mwSize)M);
		mxSetPr(tdataref,(double*)matrix);

		//transpose
		mexCallMATLAB(1,&dataref,1,&tdataref, "transpose");
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,int* matrix, int M,int N){{{1*/
void WriteMatlabData(mxArray** pdataref,int* matrix, int M,int N){

	mxArray* dataref=NULL;
	mxArray* tdataref=NULL;

	if(matrix){

		/*convert to double matrix*/
		double* doublematrix=(double*)xmalloc(M*N*sizeof(double));
		for(int i=0;i<M*N;i++) doublematrix[i]=(double)matrix[i];

		/*data is a double* pointer. Copy into a matrix: */
		tdataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(tdataref,(mwSize)N);
		mxSetN(tdataref,(mwSize)M);
		mxSetPr(tdataref,(double*)doublematrix);

		//transpose
		mexCallMATLAB(1,&dataref,1,&tdataref, "transpose");
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,Vec vector){{{1*/
void WriteMatlabData(mxArray** pdataref,Vec vector){
	
	mxArray* dataref=NULL;
	
	if(vector){
		
		/*call toolkit routine: */
		PetscVectorToMatlabVector(&dataref,vector);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;

}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,double* vector, int M){{{1*/
void WriteMatlabData(mxArray** pdataref,double* vector, int M){
	
	mxArray* dataref=NULL;

	if(vector){

		/*data is a double* pointer. Copy into a vector: */
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(dataref,(mwSize)M);
		mxSetN(dataref,(mwSize)1);
		mxSetPr(dataref,vector);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,double scalar){{{1*/
void WriteMatlabData(mxArray** pdataref,double scalar){

	*pdataref=mxCreateDoubleScalar(scalar);
}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,int integer){{{1*/
void WriteMatlabData(mxArray** pdataref,int integer){

		*pdataref=mxCreateDoubleScalar((double)integer);

}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,int boolean){{{1*/
void WriteMatlabData(mxArray** pdataref,bool boolean){

	*pdataref=mxCreateDoubleScalar((double)boolean);

}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,char* string){{{1*/
void WriteMatlabData(mxArray** pdataref,char* string){

		*pdataref=mxCreateString(string);
}
/*}}}*/
/*FUNCTION WriteMatlabData(mxArray** pdataref,Parameters* parameters){{{1*/
void WriteMatlabData(mxArray** pdataref,Parameters* parameters){

	int i;

	/*output: */
	mxArray* dataref=NULL;
	mwSize nfields;
	const	char**	fnames=NULL;
	mwSize		onebyone[2] = {1,1};
	mwSize		ndim=2;

	/*intermediary: */
	Param*      param=NULL;

	/*Recover data from the parameters dataset: */
	nfields=(mwSize)parameters->Size();
	fnames=(const char**)xmalloc(nfields*sizeof(char*));
	
	/*Build structure in matlab workspace with all the parameter fields: */
	for(i=0;i<nfields;i++){
		param=(Param*)parameters->GetObjectByOffset(i);
		fnames[i]=param->GetParameterName();
	}
	/*Initialize structure: */
	dataref=mxCreateStructArray( ndim,onebyone,nfields,fnames);

	/*Fill each field: */
	for(i=0;i<nfields;i++){

		param=(Param*)parameters->GetObjectByOffset(i);
		param->SetMatlabField(dataref);
	}
		
	/*Assign output pointers:*/
	*pdataref=dataref;

}
/*}}}*/
#endif
