%
%  plot a relative histogram and cdf optionally along with
%  a normal distribution.
%
%  []=plot_hist_norm(dresp1   ,dresp2  ,hmin,hmax,hnint)
%  []=plot_hist_norm(samp,desc,mu,sigma,hmin,hmax,hnint)
%
%  where the required input is:
%    dresp1        (structure array, responses)
%      or
%    samp          (double array, lists of samples)
%    desc          (cell array, list of descriptions)
%
%  and the optional input is:
%    dresp2        (structure array, responses)
%      or
%    mu            (double vector, means)
%    sigma         (double vector, standard deviations)
%
%    hmin          (numeric, minimum for histogram)
%    hmax          (numeric, maximum for histogram)
%    hnint         (numeric, number of intervals for histogram)
%
%  the required fields of dresp1 are:
%    descriptor    (char, description)
%    sample        (double vector, list of samples)
%
%  and the required fields of dresp2 are:
%    mean          (double, mean of sample)
%    stddev        (double, standard deviation of sample)
%
%  for each response in the input array, this function
%  calculates and plots a relative histogram and CDF of the list
%  of samples, and annotates it with the description.  in
%  addition, a mean and standard deviation may be supplied or
%  calculated so that a normal distribution and CDF may be
%  plotted.
%
%  dresp1 data would typically be contained in the dakota tabular
%  output file from a sampling analysis, and dresp2 data would
%  typically be contained in the dakota output file from a local
%  sensitivity analysis, both read by dakota_out_parse.
%
%  "Copyright 2009, by the California Institute of Technology.
%  ALL RIGHTS RESERVED. United States Government Sponsorship
%  acknowledged. Any commercial use must be negotiated with
%  the Office of Technology Transfer at the California Institute
%  of Technology.  (NTR 47078)
%
%  This software may be subject to U.S. export control laws.
%  By accepting this  software, the user agrees to comply with
%  all applicable U.S. export laws and regulations. User has the
%  responsibility to obtain export licenses, or other export
%  authority as may be required before exporting such information
%  to foreign countries or providing access to foreign persons."
%
function []=plot_hist_norm(varargin)

if ~nargin
    help plot_hist_norm
    return
end

%%  process input data and assemble into matrices as needed

ivar=1;
if isstruct(varargin{ivar})
    dresp1=varargin{ivar};
    ivar=ivar+1;
    
    desc=cell (1,length(dresp1));
    lsamp=zeros(1,length(dresp1));
    for i=1:length(dresp1)
        lsamp(i)=length(dresp1(i).sample);
    end
    samp=zeros(max(lsamp),length(dresp1));
    samp(:,:)=NaN;

    for i=1:length(dresp1)
        desc(i)=cellstr(dresp1(i).descriptor);
        samp(1:lsamp(i),i)=dresp1(i).sample;
    end
else
    samp=varargin{ivar};
    ivar=ivar+1;
    
    lsamp(1:size(samp,2))=size(samp,1);

    if ivar <= nargin && iscell(varargin{ivar})
        desc=varargin{ivar};
        ivar=ivar+1;
    else
        desc={};
    end 
end

if     ivar <= nargin && isstruct(varargin{ivar})
    dresp2=varargin{ivar};
    ivar=ivar+1;

    mu   =zeros(1,length(dresp2));
    sigma=zeros(1,length(dresp2));

    for i=1:length(dresp2)
        mu   (i)=dresp2(i).mean;
        sigma(i)=dresp2(i).stddev;
    end
elseif ivar+1 <= nargin
    if isnumeric(varargin{ivar})
        if ~isempty(varargin{ivar})
            mu   =varargin{ivar};
        else
            mu   =mean(samp);
            display('Using calculated means.')
        end
    end
    ivar=ivar+1;
    if isnumeric(varargin{ivar})
        if ~isempty(varargin{ivar})
            sigma=varargin{ivar};
        else
            sigma=std(samp);
            display('Using calculated standard deviations.')
        end
    end
    ivar=ivar+1;
end

%%  generate the intervals

if ivar <= nargin && ~isempty(varargin{ivar})
    hmin=varargin{ivar};
else
    hmin=min(min(samp));
end
ivar=ivar+1;
if ivar <= nargin && ~isempty(varargin{ivar})
    hmax=varargin{ivar};
else
    hmax=max(max(samp));
end
ivar=ivar+1;
if ivar <= nargin && ~isempty(varargin{ivar})
    hnint=varargin{ivar};
else
    hnint=50;
end
ivar=ivar+1;
edges=hmin:(hmax-hmin)/hnint:hmax;

%%  generate the histogram counts and make them relative

%  note that for the histc function:
%  n(k) counts the value x(i) if edges(k) <= x(i) < edges(k+1).
%  The last bin counts any values of x that match edges(end).
%  Values outside the values in edges are not counted.
%  Use -inf and inf in edges to include all non-NaN values.

dhistc=histc(samp,edges);
for i=1:size(samp,2)
    dbelow(i)  =length(find(samp(:,i)<edges(  1)))/lsamp(i);
    dhistc(:,i)=dhistc(:,i)                       /lsamp(i);
    dabove(i)  =length(find(samp(:,i)>edges(end)))/lsamp(i);
end

if exist('mu','var') && exist('sigma','var')
    ncol=size(samp,2);
    for i=1:ncol
        dbelow(ncol+i)=normcdf(edges(  1),mu(i),sigma(i));
        dhistc(1:size(dhistc,1)-1,ncol+i)=...
            normcdf(edges(2:end  ),mu(i),sigma(i))-...
            normcdf(edges(1:end-1),mu(i),sigma(i));
        dabove(ncol+i)=norminv(edges(end),mu(i),sigma(i));
        if exist('desc','var')
            desc(ncol+i)={[desc{i} ' norm']};
        end
    end
end

%  draw the bar plot

hl1=bar(edges(1:end-1),dhistc(1:end-1,:));
ax1=gca;

%  add the annotation

title('Relative Frequency Histogram')
xlabel('Interval Edge Value')
ylabel('Relative Frequency')

if exist('desc','var')
    hleg1=legend(ax1,desc,'Location','NorthWest',...
                 'Interpreter','none');
else
    hleg1=legend(ax1);
end

%%  generate the cumulative distribution functions

% cdf=zeros(size(dhistc));
% cdf(1,:)=dhistc(1,:);
% for i=2:size(dhistc,1)
%     cdf(i,:)=cdf(i-1,:)+dhistc(i,:);
% end
cdf=cumsum(dhistc);
for i=1:size(dhistc,2)
    cdf(:,i)=dbelow(i)+cdf(:,i);
end
if exist('desc','var')
    ncol=length(desc);
    for i=1:ncol
        cdesc(i)={[desc{i} ' cdf']};
    end
end

%  draw the line plot

%  (see "Using Multiple X- and Y-Axes" and "Overlaying Other
%  Plots on Bar Graphs", or search on "YAxisLocation right")

%hold all
%hold on
%plot(edges,cdf)
%plotyy([],[],edges,cdf)

%  ticks from the bar plot will show through, so make ticks
%  from the line plot match

nytick=length(get(ax1,'YTick'));
ytmax =ceil(max(max(cdf))/0.1-0.1)*0.1;
ytinc =ytmax/(nytick-1-1);

ax2=axes('Position',get(ax1,'Position'),...
         'XLim',get(ax1,'XLim'),...
         'YLim',[0 ytinc*(nytick-1)],...
         'YTick',[0:ytinc:ytinc*(nytick-1)],...
         'XAxisLocation','bottom','YAxisLocation','right',...
         'Color','none','Layer','top');
hl2=line(edges(1:end-1),cdf(1:end-1,:),'Parent',ax2);
       
%  add the annotation

ylabel('Cumulative Percent')

% legend doesn't combine with bar chart above
if exist('cdesc','var')
    hleg2=legend(ax2,cdesc,'Location','NorthEast',...
                 'Interpreter','none');
    set(hleg2,'Color','white')
else
    hleg2=legend(ax2);
    set(hleg2,'Color','white')
end

set(gcf,'PaperPositionMode','auto')
%hold off

end
