%
%  plot a stacked bar chart of the sample distributions.
%
%  []=plot_sampdist_bars(dresp,lstr)
%
%  where the required input is:
%    dresp         (structure array, responses)
%
%  and the optional input is:
%    lstr          (cell array, legend labels)
%
%  the required fields of dresp are:
%    descriptor    (char, description)
%    sample        (double vector, list of samples)
%
%  and the optional fields of dresp are:
%    min           (double, minimum of sample)
%    quart1        (double, first quartile of sample)
%    median        (double, median of sample)
%    quart3        (double, third quartile of sample)
%    max           (double, maximum of sample)
%
%  for each response in the input array, this function plots
%  a stacked bar plot of the list of samples, where the bars
%  are stacked by the four quartiles, and annotates it with
%  the description.  the quartiles will be calculated from the
%  samples if they do not already exist.
%
%  this data would typically be contained in the dakota tabular
%  output file and read by dakota_out_parse.
%
%  "Copyright 2009, by the California Institute of Technology.
%  ALL RIGHTS RESERVED. United States Government Sponsorship
%  acknowledged. Any commercial use must be negotiated with
%  the Office of Technology Transfer at the California Institute
%  of Technology.  (NTR 47078)
%
%  This software may be subject to U.S. export control laws.
%  By accepting this  software, the user agrees to comply with
%  all applicable U.S. export laws and regulations. User has the
%  responsibility to obtain export licenses, or other export
%  authority as may be required before exporting such information
%  to foreign countries or providing access to foreign persons."
%
function []=plot_sampdist_bars(dresp,lstr)

if ~nargin
    help plot_sampdist_bars
    return
end

%%  assemble the data into a matrix and calculate the increments

desc=cell (1,length(dresp));
data=zeros(length(dresp),5);

if ~isfield(dresp,'min')    || ~isfield(dresp,'quart1') || ...
   ~isfield(dresp,'median') || ~isfield(dresp,'quart3') || ...
   ~isfield(dresp,'max')
    for i=1:length(dresp)
        dresp(i).min   =min    (dresp(i).sample);
        dresp(i).quart1=prctile(dresp(i).sample,25);
        dresp(i).median=median (dresp(i).sample);
        dresp(i).quart3=prctile(dresp(i).sample,75);
        dresp(i).max   =max    (dresp(i).sample);
    end
end

for i=1:length(dresp)
    desc(i)=cellstr(dresp(i).descriptor);
    data(i,1)=dresp(i).min;
    data(i,2)=dresp(i).quart1-dresp(i).min;
    data(i,3)=dresp(i).median-dresp(i).quart1;
    data(i,4)=dresp(i).quart3-dresp(i).median;
    data(i,5)=dresp(i).max   -dresp(i).quart3;
end

%%  draw the stacked bar plot

%  if there's only one row, Matlab 7.5 interprets it as a column,
%  so add an extra row, then reduce xlim

if length(dresp) == 1
    data=[data; data];
end

hl1=bar(data,'stacked');
%  set barseries properties for lowest value
whitebg('white')
set(hl1(1),'FaceColor','white')
set(hl1(1),'Visible','off')

ax1=gca;
if length(dresp) == 1
    set(ax1,'xlim',[0.5 1.5])
end
set(ax1,'xtick',1:1:max(length(dresp),2));
set(ax1,'xticklabel',desc)

%  add the annotation

title('Sample Distributions of Variables and/or Responses')
xlabel('Variable or Response')
ylabel('Value')

if ~exist('lstr','var') || isempty(lstr)
    lstr={'minimum' 'quartile 1' 'median' 'quartile 3' 'maximum'};
end

hleg1=legend(ax1,lstr,'Location','EastOutside',...
             'Orientation','vertical','Interpreter','none');

end
