%
%  plot a stacked bar chart of the response levels in the cdf.
%
%  []=plot_rlev_bars(dresp,lstr)
%
%  where the required input is:
%    dresp         (structure array, responses)
%
%  and the optional input is:
%    lstr          (cell array, legend labels)
%
%  the required fields of dresp are:
%    descriptor    (char, description)
%    cdf(:,4)      (double matrix, CDF table)
%
%  for each response in the input array, this function plots
%  a stacked bar plot of the responses, where the bars are
%  stacked by the response levels corresponding to the given
%  probabilities in the CDF, and annotates it with the
%  description.  the legend labels can be given or constructed
%  from the probabilities.
%
%  this data would typically be contained in the dakota output
%  file and read by dakota_out_parse.
%
%  "Copyright 2009, by the California Institute of Technology.
%  ALL RIGHTS RESERVED. United States Government Sponsorship
%  acknowledged. Any commercial use must be negotiated with
%  the Office of Technology Transfer at the California Institute
%  of Technology.  (NTR 47078)
%
%  This software may be subject to U.S. export control laws.
%  By accepting this  software, the user agrees to comply with
%  all applicable U.S. export laws and regulations. User has the
%  responsibility to obtain export licenses, or other export
%  authority as may be required before exporting such information
%  to foreign countries or providing access to foreign persons."
%
function []=plot_rlev_bars(dresp,lstr)

if ~nargin
    help plot_rlev_bars
    return
end

%%  assemble the data into a matrix and calculate the increments

desc=cell (1,length(dresp));
for i=1:length(dresp)
    ldata(i)=size(dresp(i).cdf,1);
end
data=zeros(length(dresp),max(ldata));

for i=1:length(dresp)
    desc(i)=cellstr(dresp(i).descriptor);
    if ~isempty(dresp(i).cdf)
        data(i,1)=dresp(i).cdf(1,1);
        for j=2:size(dresp(i).cdf,1)
            if (dresp(i).cdf(j,1) > dresp(i).cdf(j-1,1))
                data(i,j)=dresp(i).cdf(j,1)-dresp(i).cdf(j-1,1);
            end
        end
    end
end

%%  draw the stacked bar plot

%  if there's only one row, Matlab 7.5 interprets it as a column,
%  so add an extra row, then reduce xlim

if length(dresp) == 1
    data=[data; data];
end

hl1=bar(data,'stacked');
%  set barseries properties for lowest value
whitebg('white')
set(hl1(1),'FaceColor','white')
set(hl1(1),'Visible','off')

ax1=gca;
if length(dresp) == 1
    set(ax1,'xlim',[0.5 1.5])
end
set(ax1,'xticklabel',desc)

%  add the annotation

title('Response Levels for Specified Probabilities (PMA)')
xlabel('Response')
ylabel('Response Level')

if ~exist('lstr','var') || isempty(lstr)
    lstr=cell(1,max(ldata));
    for i=1:max(ldata)
        lstr(i)=cellstr(sprintf('%g%%',...
            100*dresp(find(ldata == max(ldata),1,'first')).cdf(i,2)));
    end
    if ~isempty(find(ldata < max(ldata)))
        warning('Variable number of probabilities for responses.');
    end
end

hleg1=legend(ax1,lstr,'Location','EastOutside',...
             'Orientation','vertical','Interpreter','none');

end
