%
%  plot a stacked bar chart of the probabilities in the CDF.
%
%  []=plot_prob_bars(dresp,lstr)
%
%  where the required input is:
%    dresp         (structure array, responses)
%
%  and the optional input is:
%    lstr          (cell array, legend labels)
%
%  the required fields of dresp are:
%    descriptor    (char, description)
%    cdf(:,4)      (double matrix, CDF table)
%
%  for each response in the input array, this function plots
%  a stacked bar plot of the responses, where the bars are
%  stacked by the probabilities corresponding to the given
%  response levels in the CDF, and annotates it with the
%  description.  the legend labels can be given or constructed
%  from the response levels.
%
%  this data would typically be contained in the dakota output
%  file and read by dakota_out_parse.
%
%  "Copyright 2009, by the California Institute of Technology.
%  ALL RIGHTS RESERVED. United States Government Sponsorship
%  acknowledged. Any commercial use must be negotiated with
%  the Office of Technology Transfer at the California Institute
%  of Technology.  (NTR 47078)
%
%  This software may be subject to U.S. export control laws.
%  By accepting this  software, the user agrees to comply with
%  all applicable U.S. export laws and regulations. User has the
%  responsibility to obtain export licenses, or other export
%  authority as may be required before exporting such information
%  to foreign countries or providing access to foreign persons."
%
function []=plot_prob_bars(dresp,lstr)

if ~nargin
    help plot_prob_bars
    return
end

%%  assemble the data into a matrix and calculate the increments

desc=cell (1,length(dresp));
for i=1:length(dresp)
    ldata(i)=size(dresp(i).cdf,1);
end
data=zeros(length(dresp),max(ldata));

for i=1:length(dresp)
    desc(i)=cellstr(dresp(i).descriptor);
    if ~isempty(dresp(i).cdf)
    data(i,1)=dresp(i).cdf(1,2);
        for j=2:size(dresp(i).cdf,1)
            if (dresp(i).cdf(j,2) > dresp(i).cdf(j-1,2))
                data(i,j)=dresp(i).cdf(j,2)-dresp(i).cdf(j-1,2);
            end
        end
    end
end

%  convert to percentage

data=data*100.;

%%  draw the stacked bar plot

%  if there's only one row, Matlab 7.5 interprets it as a column,
%  so add an extra row, then reduce xlim

if length(dresp) == 1
    data=[data; data];
end

hl1=bar(data,'stacked');
% set(hl1(1),'FaceColor','green')
% set(hl1(2),'FaceColor','blue')
% set(hl1(3),'FaceColor','yellow')
% set(hl1(4),'FaceColor','cyan')
% set(hl1(5),'FaceColor','magenta')
% set(hl1(6),'FaceColor','red')

ax1=gca;
set(ax1,'ylim',[0 120])
if length(dresp) == 1
    set(ax1,'xlim',[0.5 1.5])
end
set(ax1,'xticklabel',desc)

%  add the annotation

title('Probabilities for Specified Response Levels (RIA)')
xlabel('Response')
ylabel('Percent Below Level')

if ~exist('lstr','var') || isempty(lstr)
    lstr=cell(1,max(ldata));
    for i=1:max(ldata)
        lstr(i)=cellstr(sprintf('%g',...
            dresp(find(ldata == max(ldata),1,'first')).cdf(i,1)));
    end
    if ~isempty(find(ldata < max(ldata)))
        warning('Variable number of levels for responses.');
    end
end

hleg1=legend(ax1,lstr,'Location','EastOutside',...
             'Orientation','vertical','Interpreter','none');

end
