%
%  plot a stacked bar chart of the sample distributions.
%
%  []=plot_normdist_bars(dresp,plev,lstr)
%
%  where the required input is:
%    dresp         (structure array, responses)
%
%  and the optional input is:
%    plev          (double vector, probability levels)
%    lstr          (cell array, legend labels)
%
%  the required fields of dresp are:
%    descriptor    (char, description)
%    sample        (double vector, list of samples)
%
%  and the optional fields of dresp are:
%    mean          (double, mean of sample)
%    stddev        (double, standard deviation of sample)
%
%  for each response in the input array, this function plots
%  a stacked bar plot of the list of samples, where the bars
%  are stacked by the given or default probability levels
%  calculated from a normal distribution, and annotates it with
%  the description.  the mean and standard deviation will be
%  calculated from the samples if they do not already exist.
%  the legend labels can be given or constructed from the
%  probability levels.
%
%  this data would typically be contained in the dakota tabular
%  output file and read by dakota_out_parse.
%
%  "Copyright 2009, by the California Institute of Technology.
%  ALL RIGHTS RESERVED. United States Government Sponsorship
%  acknowledged. Any commercial use must be negotiated with
%  the Office of Technology Transfer at the California Institute
%  of Technology.  (NTR 47078)
%
%  This software may be subject to U.S. export control laws.
%  By accepting this  software, the user agrees to comply with
%  all applicable U.S. export laws and regulations. User has the
%  responsibility to obtain export licenses, or other export
%  authority as may be required before exporting such information
%  to foreign countries or providing access to foreign persons."
%
function []=plot_normdist_bars(dresp,plev,lstr)

if ~nargin
    help plot_normdist_bars
    return
end

if ~exist('plev','var') || isempty(plev)
    plev=[0.01 0.25 0.50 0.75 0.99];
end

%%  assemble the data into a matrix and calculate the increments

desc=cell (1,length(dresp));
data=zeros(length(dresp),5);

if ~isfield(dresp,'mean') || ~isfield(dresp,'stddev')
    for i=1:length(dresp)
%  calculate 95% confidence intervals (same as dakota)
        [dresp(i).mean,dresp(i).stddev,...
            dresp(i).meanci,dresp(i).stddevci]=...
            normfit(dresp(i).sample,0.05);
    end
end

for i=1:length(dresp)
    desc(i)=cellstr(dresp(i).descriptor);
    data(i,:)=norminv(plev,dresp(i).mean,dresp(i).stddev);
end

for j=length(plev):-1:2
    data(:,j)=data(:,j)-data(:,j-1);
end

%%  draw the stacked bar plot

%  if there's only one row, Matlab 7.5 interprets it as a column,
%  so add an extra row, then reduce xlim

if length(dresp) == 1
    data=[data; data];
end

hl1=bar(data,'stacked');
%  set barseries properties for lowest value
whitebg('white')
set(hl1(1),'FaceColor','white')
set(hl1(1),'Visible','off')

ax1=gca;
if length(dresp) == 1
    set(ax1,'xlim',[0.5 1.5])
end
set(ax1,'xtick',1:1:max(length(dresp),2));
set(ax1,'xticklabel',desc)

%  add the annotation

title('Normal Distributions of Variables and/or Responses')
xlabel('Variable or Response')
ylabel('Value')

if ~exist('lstr','var') || isempty(lstr)
    lstr=cell(1,length(plev));
    for i=1:length(plev)
        lstr(i)=cellstr(sprintf('%g%%',100*plev(i)));
    end
end

hleg1=legend(ax1,lstr,'Location','EastOutside',...
             'Orientation','vertical','Interpreter','none');

end
