%
%  plot a stacked bar chart of the importance factors.
%
%  []=plot_if_bars(dresp,ifmin,isort)
%
%  where the required input is:
%    dresp         (structure array, responses)
%
%  and the optional input is:
%    ifmin         (double, minimum importance factor)
%    isort         (numeric, sort flag:  0, no sorting;
%                                        1, highest at bottom;
%                                       -1, lowest at bottom)
%
%  the required fields of dresp are:
%    descriptor    (char, description)
%    var           (cell array, variables)
%    impfac        (double array, importance factors)
%
%  for each response in the input array, this function plots
%  a stacked bar plot of the responses, where the bars are
%  stacked by the importance factors, and annotates it with the
%  description.  the legend labels are constructed from the
%  variable list.
%
%  this data would typically be contained in the dakota output
%  file from a local reliability analysis and read by
%  dakota_out_parse.
%
%  "Copyright 2009, by the California Institute of Technology.
%  ALL RIGHTS RESERVED. United States Government Sponsorship
%  acknowledged. Any commercial use must be negotiated with
%  the Office of Technology Transfer at the California Institute
%  of Technology.  (NTR 47078)
%
%  This software may be subject to U.S. export control laws.
%  By accepting this  software, the user agrees to comply with
%  all applicable U.S. export laws and regulations. User has the
%  responsibility to obtain export licenses, or other export
%  authority as may be required before exporting such information
%  to foreign countries or providing access to foreign persons."
%
function []=plot_if_bars(dresp,ifmin,isort)

if ~nargin
    help plot_if_bars
    return
end

if ~exist('ifmin','var') || isempty(ifmin)
    ifmin=0;
end

if ~exist('isort','var') || isempty(isort)
    isort=0;
end

%%  assemble the data into a matrix

desc=cell (1,length(dresp));
for i=1:length(dresp)
    ldata(i)=length(dresp(i).impfac);
end
data=zeros(length(dresp),max(ldata));
dvar=dresp(find(ldata == max(ldata),1,'first')).var;

for i=1:length(dresp)
    desc(i)=cellstr(dresp(i).descriptor);
    data(i,1:ldata(i))=dresp(i).impfac;
end

%%  sort the data, if necessary

if (isort)
    ifmean=mean(data,1);
    if (isort > 0)
        [ifmean,index]=sort(ifmean,'descend');
    else
        [ifmean,index]=sort(ifmean,'ascend' );
    end
    clear ifmean
    
    dvar=dvar(index);
    data=data(:,index);
end

%%  filter the data, if necessary

if (ifmin > 0)
    nif=length(dvar);
    dvar(nif+1,1)=cellstr(sprintf('others < %f',ifmin));
    data(:,nif+1)=0.;
    
    nif2=0;
    dvar2=cell (size(dvar));
    data2=zeros(size(data));
    
%  sum filtered rows and copy unfiltered rows

    for i=1:nif
        if (max(data(:,i)) < ifmin)
            data(:,nif+1)=data(:,nif+1)+data(:,i);
        else
            nif2=nif2+1;
            dvar2(nif2)  =dvar(i);
            data2(:,nif2)=data(:,i);
        end
    end
    
%  copy sums

    dvar2(nif2+1)  =dvar(nif+1);
    data2(:,nif2+1)=data(:,nif+1);
    
%  copy back and truncate filtered rows

    clear dvar data
    if (isort >= 0)
        dvar(1:nif2+1)  =dvar2(1:nif2+1);
        data(:,1:nif2+1)=data2(:,1:nif2+1);
    else
        dvar(1       )  =dvar2(  nif2+1);
        dvar(2:nif2+1)  =dvar2(1:nif2  );
        data(:,1       )=data2(:,  nif2+1);
        data(:,2:nif2+1)=data2(:,1:nif2  );
    end
    clear nif nif2 dvar2 data2
end

%%  draw the stacked bar plot

%  if there's only one row, Matlab 7.5 interprets it as a column,
%  so add an extra row, then reduce xlim

if length(dresp) == 1
    data=[data; data];
end

hl1=bar(data,'stacked');

ax1=gca;
set(ax1,'ylim',[0 1.2])
if length(dresp) == 1
    set(ax1,'xlim',[0.5 1.5])
end
set(ax1,'xticklabel',desc)

%  add the annotation

title('Importance Factors')
xlabel('Response')
ylabel('Importance Factor Value')

hleg1=legend(ax1,dvar,'Location','EastOutside',...
             'Orientation','vertical','Interpreter','none');

end
